/*
 * Creator: Naman Dixit
 * Notice: © Copyright 2020 Naman Dixit
 */

#include "nlib/nlib.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <netinet/in.h>
#include <sys/epoll.h>
#include <sys/time.h>
#include <assert.h>
#include <signal.h>
#include <librdkafka/rdkafka.h>

# if defined(COMPILER_CLANG)
#  pragma clang diagnostic push
#   pragma clang diagnostic ignored "-Wpadded"
#   pragma clang diagnostic ignored "-Wfloat-equal"
# endif
#include "cJSON/cJSON.h"
#include "cJSON/cJSON.c"
# if defined(COMPILER_CLANG)
#  pragma clang diagnostic pop
# endif

typedef struct Resources {
    Sint memory;
} Resources;

typedef struct Command {
    Char *txn_id;
    Resources res;
} Command;

#include "kafka.h"
#include "time.c"

global_variable volatile sig_atomic_t global_keep_running = 1;

internal_function
void signalHandlerSIGINT (int _)
{
    (void)_;
    global_keep_running = 0;
}

int main(int argc, char** argv)
{
    unused_variable(argc);
    unused_variable(argv);

    signal(SIGINT, signalHandlerSIGINT);

    Kafka kafka = {0};

    kafka.writer = kafkaCreateWriter(&kafka, "10.129.6.5:9092");
    kafka.reader = kafkaCreateReader(&kafka, "10.129.6.5:9092");

    rd_kafka_topic_partition_list_t *kafka_reader_topics = rd_kafka_topic_partition_list_new(1);

    rd_kafka_topic_t *topic_req_a2g = kafkaSubscribe(&kafka, kafka_reader_topics,
                                                     "REQUEST_ARBITER_2_GRUNT");
    unused_variable(topic_req_a2g);

    rd_kafka_resp_err_t kafka_reader_topics_err = rd_kafka_subscribe(kafka.reader, kafka_reader_topics);
    rd_kafka_topic_partition_list_destroy(kafka_reader_topics);

    if (kafka_reader_topics_err) {
        fprintf(stderr, "Subscribe failed: %s\n",
                rd_kafka_err2str(kafka_reader_topics_err));
        rd_kafka_destroy(kafka.reader);
        return -1;
    }

    Char *join_msg = NULL;
    sbufPrint(join_msg, "{\"id\": \"my-machine\"");
    sbufPrint(join_msg, "\n}\n");

    if (!kafkaWrite(kafka.writer, "JOIN_GRUNT_2_ARBITER", "rm_grunt", join_msg)) {
        return -1;
    }

    U64 time_begin = timeMilli();
    U64 time_accum = 0;

    while (global_keep_running) {
        // NOTE(naman): Get the fd's that are ready
        rd_kafka_message_t *kafka_message_read = rd_kafka_consumer_poll(kafka.reader, 100);

        B32 command_found = false;
        Command c = {0};

        if (kafka_message_read != NULL) {
            if (kafka_message_read->err) {
                /* Consumer error: typically just informational. */
                fprintf(stderr, "Consumer error: %s\n",
                        rd_kafka_message_errstr(kafka_message_read));
            } else {
                fprintf(stderr,
                        "Received message on %s [%d] "
                        "at offset %"PRId64": \n%s\n",
                        rd_kafka_topic_name(kafka_message_read->rkt),
                        (int)kafka_message_read->partition, kafka_message_read->offset,
                        cJSON_Print(cJSON_Parse((char *)kafka_message_read->payload)));

                char *buffer = (char *)kafka_message_read->payload;

                const Char *json_error = NULL;
                cJSON *root = cJSON_ParseWithOpts(buffer, &json_error, true);

                if (root == NULL) {
                    // TODO(naman): Error
                } else {
                    command_found = true;
                    c.txn_id = cJSON_GetObjectItem(root, "id")->valuestring;
                    c.res.memory = cJSON_GetObjectItem(root, "memory")->valueint;
                }
            }
            rd_kafka_message_destroy(kafka_message_read);
        }

        int memory = 0;

        FILE *meminfo = fopen("/proc/meminfo", "r");
        Char line[256] = {0};
        while(fgets(line, sizeof(line), meminfo)) {
            if (sscanf(line, "MemAvailable: %d kB", &memory) == 1) {
                fclose(meminfo);
                break;
            }
        }
        memory /= 1024;

        if (command_found) {
            Char *output = NULL;

            sbufPrint(output, "{\n\"id\": \"%s\"", c.txn_id);

            if (memory >= c.res.memory) {
                sbufPrint(output, ",\n\"success\": %d\n", 1);
                // TODO(naman): Add port
                // sbufPrint(output, ",\n\"port\": %d\n", port);
            } else {
                sbufPrint(output, ",\n\"success\": %d\n", 0);
            }

            sbufPrint(output, "\n}\n");

            if (!kafkaWrite(kafka.writer, "RESPONSE_GRUNT_2_ARBITER", "rm_grunt", output)) {
                return -1;
            }
        } else { // Send a heartbeat message if it is time to do so
            U64 time_new = timeMilli();
            U64 time_passed = time_new - time_begin;
            time_begin = time_new;
            time_accum += time_passed;

            if (time_accum >= 1000) {
                time_accum = 0;

                Char *output = NULL;

                sbufPrint(output, "{\"id\": \"my-machine\"");
                sbufPrint(output, ",\n\"memory\": %d", memory);

                sbufPrint(output, "\n}\n");

                if (!kafkaWrite(kafka.writer, "HEARTBEAT_GRUNT_2_ARBITER", "rm_grunt", output)) {
                    return -1;
                }
            }
        }
    }

    for (Size i = 0; i < sbufElemin(kafka.topics); i++) {
        rd_kafka_topic_destroy(kafka.topics[i]);
    }
    rd_kafka_consumer_close(kafka.reader);
    rd_kafka_destroy(kafka.reader);
    for (Size i = 0; i < sbufElemin(kafka.queues); i++) {
        rd_kafka_queue_destroy(kafka.queues[i]);
    }
    rd_kafka_destroy(kafka.writer);

    return 0;
}
