/*
 * Creator: Naman Dixit
 * Notice: © Copyright 2020 Naman Dixit
 */

#define logMessage(s, ...) printf(s "\n", ##__VA_ARGS__)

#include "nlib/nlib.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/epoll.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/time.h>
#include <arpa/inet.h>
#include <assert.h>
#include <signal.h>
#include <librdkafka/rdkafka.h>

typedef struct Grunt {
    Char *id;
    Sint memory;
} Grunt;

typedef struct Grunt_Survey {
    Grunt **grunt_ptrs;
    U16 *ports;
    U64 milli_passed;
    U64 milli_last;
    Char *txn_id;
} Grunt_Survey;

typedef struct Command {
    enum Command_Kind {
        Command_NONE,
        Command_REQUEST_DISPATCHER_2_ARBITER,
        Command_RESPONSE_ARBITER_2_DISPATCHER,

        Command_REQUEST_ARBITER_2_GRUNT,
        Command_RESPONSE_GRUNT_2_ARBITER,

        Command_HEARTBEAT_GRUNT_2_ARBITER,
    } kind;

    Char *txn_id;

    union {
        struct {
            Sint memory;
        } req_d2a;

        struct {
            Char **grunt_ids;
        } res_a2d;

        struct {
            Sint memory;
        } req_a2g;

        struct {
            Char *id;
            U16 port;
        } res_g2a;

        struct {
            Char *id;
            Sint memory;
        } beat_g2a;
    };
} Command;

# if defined(COMPILER_CLANG)
#  pragma clang diagnostic push
#   pragma clang diagnostic ignored "-Wpadded"
#   pragma clang diagnostic ignored "-Wfloat-equal"
# endif
#include "cJSON/cJSON.h"
#include "cJSON/cJSON.c"
# if defined(COMPILER_CLANG)
#  pragma clang diagnostic pop
# endif

#include "kafka.h"
#include "time.c"

global_variable volatile sig_atomic_t global_keep_running = 1;

internal_function
void signalHandlerSIGINT (int _)
{
    (void)_;
    global_keep_running = 0;
}

Sint main (Sint argc, Char *argv[])
{
    unused_variable(argc);
    unused_variable(argv);

    signal(SIGINT, signalHandlerSIGINT);

    Command *commands = NULL;
    Grunt *grunts = NULL;
    Hash_Table grunt_map = htCreate(0);
    Hash_Table grunt_survey_map = htCreate(0);

    sbufAdd(grunts, (Grunt){0}); // SInce 0 index out of hash table will be invalid

    Kafka kafka = {0};

    kafkaCreateWriter(&kafka, "10.129.6.5:9092");
    kafkaCreateReader(&kafka, "10.129.6.5:9092");

    rd_kafka_topic_partition_list_t *kafka_reader_topics = rd_kafka_topic_partition_list_new(1);

    rd_kafka_topic_t *topic_req_d2a = kafkaSubscribe(&kafka, kafka_reader_topics,
                                                     "REQUEST_DISPATCHER_2_ARBITER");
    rd_kafka_topic_t *topic_join_g2a = kafkaSubscribe(&kafka, kafka_reader_topics,
                                                      "JOIN_GRUNT_2_ARBITER");
    rd_kafka_topic_t *topic_res_g2a = kafkaSubscribe(&kafka, kafka_reader_topics,
                                                     "RESPONSE_GRUNT_2_ARBITER");
    rd_kafka_topic_t *topic_beat_g2a = kafkaSubscribe(&kafka, kafka_reader_topics,
                                                      "HEARTBEAT_GRUNT_2_ARBITER");

    rd_kafka_resp_err_t kafka_reader_topics_err = rd_kafka_subscribe(kafka.reader,
                                                                     kafka_reader_topics);
    rd_kafka_topic_partition_list_destroy(kafka_reader_topics);

    if (kafka_reader_topics_err) {
        fprintf(stderr, "Subscribe failed: %s\n",
                rd_kafka_err2str(kafka_reader_topics_err));
        rd_kafka_destroy(kafka.reader);
        return -1;
    }

    while (global_keep_running) {
        // NOTE(naman): Get the fd's that are ready
        rd_kafka_message_t *kafka_message_read = rd_kafka_consumer_poll(kafka.reader, 100);

        if (kafka_message_read != NULL) {
            if (kafka_message_read->err) {
                /* Consumer error: typically just informational. */
                fprintf(stderr, "Consumer error: %s\n",
                        rd_kafka_message_errstr(kafka_message_read));
            } else {
                /* Proper message */
                /* fprintf(stderr, */
                /*         "Received message on %s [%d] " */
                /*         "at offset %"PRId64": \n%s\n", */
                /*         rd_kafka_topic_name(kafka_message_read->rkt), */
                /*         (int)kafka_message_read->partition, kafka_message_read->offset, */
                /*         cJSON_Print(cJSON_Parse((char *)kafka_message_read->payload))); */

                const char *json_error = NULL;
                cJSON *root = cJSON_ParseWithOpts(kafka_message_read->payload, &json_error, true);

                if (kafka_message_read->rkt == topic_req_d2a) {
                    Command c = {.kind = Command_REQUEST_ARBITER_2_GRUNT};
                    c.txn_id = cJSON_GetObjectItem(root, "id")->valuestring;

                    Sint memory = cJSON_GetObjectItem(root, "memory")->valueint;

                    logMessage("Request D2A:\tid: %s = ([memory] = %d)", c.txn_id, memory);

                    Char **grunt_ids = NULL;

                    for (Size j = 0; j < sbufElemin(grunts); j++) {
                        Grunt g = grunts[j];
                        if (g.memory >= memory) {
                            c.kind = Command_RESPONSE_ARBITER_2_DISPATCHER;
                            sbufAdd(grunt_ids, g.id);
                        }
                    }

                    if (c.kind == Command_REQUEST_ARBITER_2_GRUNT) {
                        c.req_a2g.memory = memory;
                        Grunt_Survey *gs = calloc(1, sizeof(*gs));
                        htInsert(&grunt_survey_map, hashString(c.txn_id), (Uptr)gs);

                        gs->milli_last = timeMilli();
                        gs->txn_id = c.txn_id;
                    } else if (c.kind == Command_RESPONSE_ARBITER_2_DISPATCHER) {
                        c.res_a2d.grunt_ids = grunt_ids;
                        sbufAdd(commands, c);
                    }
                } else if (kafka_message_read->rkt == topic_join_g2a) {
                    Char *id = cJSON_GetObjectItem(root, "id")->valuestring;
                    Grunt grunt = {.id = id};

                    logMessage("Join G2A:\tid: %s", id);

                    if (htLookup(&grunt_map, hashString(id)) == 0) {
                        sbufAdd(grunts, grunt);
                        htInsert(&grunt_map, hashString(id), sbufElemin(grunts) - 1);
                    }
                } else if (kafka_message_read->rkt == topic_res_g2a) {
                    Char *id = cJSON_GetObjectItem(root, "id")->valuestring;
                    B32 success = (B32)(cJSON_GetObjectItem(root, "success")->valueint);

                    logMessage("Response G2A:\tid: %s = %s", id, success ? "succeded" : "failed");

                    if (success) {
                        Grunt_Survey *gs = (Grunt_Survey *)htLookup(&grunt_survey_map,
                                                                    hashString(id));

                        if (gs != NULL) { // If it has not been already removed
                            Grunt *g = &grunts[htLookup(&grunt_map, hashString(id))];
                            sbufAdd(gs->grunt_ptrs, g);
                        }
                    }
                } else if (kafka_message_read->rkt == topic_beat_g2a) {
                    Char *id = cJSON_GetObjectItem(root, "id")->valuestring;

                    logMessage("Beat G2A:\tid: %s", id);

                    U64 index = htLookup(&grunt_map, hashString(id));
                    if (index != 0) { // Prevent any left over message
                        grunts[index].memory = cJSON_GetObjectItem(root, "memory")->valueint;
                    }
               } else {
                    // TODO(naman): Error
                }
            }

            rd_kafka_message_destroy(kafka_message_read);
        }

        for (Size i = 0; i < grunt_survey_map.slot_count; i++) {
            if (grunt_survey_map.keys[i] != 0) {
                Grunt_Survey *gs = (Grunt_Survey *)grunt_survey_map.values[i];

                U64 milli_new = timeMilli();
                gs->milli_passed += milli_new - gs->milli_last;
                gs->milli_last = milli_new;
                if (gs->milli_passed >= 1000) {
                    htRemove(&grunt_survey_map, hashString(gs->txn_id));

                    Command c = {.kind = Command_RESPONSE_ARBITER_2_DISPATCHER};
                    c.txn_id = gs->txn_id;

                    for (Size k = 0; k < sbufElemin(gs->grunt_ptrs); k++) {
                        sbufAdd(c.res_a2d.grunt_ids, gs->grunt_ptrs[k]->id);
                    }

                    sbufAdd(commands, c);
                }
            }
        }

        for (Size j = 0; j < sbufElemin(commands); j++) {
            Command c = commands[j];

            Char *output = NULL;
            Char *topic = NULL;

            if (c.kind == Command_REQUEST_ARBITER_2_GRUNT) {
                topic = "REQUEST_ARBITER_2_GRUNT";

                sbufPrint(output, "{\n\"id\": \"%s\"", c.txn_id);
                sbufPrint(output, ",\n\"memory\": %d\n", c.req_a2g.memory);
                sbufPrint(output, "\n}\n");
            } else if (c.kind == Command_RESPONSE_ARBITER_2_DISPATCHER) {
                topic = "RESPONSE_ARBITER_2_DISPATCHER";

                sbufPrint(output, "{\n\"id\": \"%s\"", c.txn_id);
                sbufPrint(output, ",\n\"grunts\": [");
                for (Size k = 0; k < sbufElemin(c.res_a2d.grunt_ids); k++) {
                    sbufPrint(output, "\"%s\"", c.res_a2d.grunt_ids[k]);
                    if (k < sbufElemin(c.res_a2d.grunt_ids) - 1) {
                        sbufPrint(output, ",");
                    }
                }
                sbufPrint(output, "]");
                sbufPrint(output, "\n}");
            }

            if (output != NULL) {
                printf("Sending to %s\n%s\n", topic, output);
                if (!kafkaWrite(kafka.writer, topic, "rm_arbiter", output)) {
                    return -1;
                }
            }
            sbufDelete(output);
            sbufDelete(commands);
        }
    }

    for (Size i = 0; i < sbufElemin(kafka.topics); i++) {
        rd_kafka_topic_destroy(kafka.topics[i]);
    }
    rd_kafka_consumer_close(kafka.reader);
    rd_kafka_destroy(kafka.reader);
    for (Size i = 0; i < sbufElemin(kafka.queues); i++) {
        rd_kafka_queue_destroy(kafka.queues[i]);
    }
    rd_kafka_destroy(kafka.writer);

    return 0;
}
