"use strict";

const express = require('express')
const bodyParser = require('body-parser')
const fileUpload = require('express-fileupload');
const constants = require('.././constants.json')
const fs = require('fs')
const { spawn } = require('child_process');
const morgan = require('morgan')
const mqtt = require('mqtt')
// const client = mqtt.connect('mqtt://' + constants.mqtt_url)
const app = express()
const libSupport = require('./lib')

/**
 * functionToPort maps the function and its respective port mapping
 * TODO: change this to hold a list of mappings of horizontal scaling
 */
let functionToPort = new Map(), 
    usedPort = new Map(), // TODO: remove after integration with RM
    rmQueue = new Map()

let kafka = require('kafka-node'),
    Producer = kafka.Producer,
    KeyedMessage = kafka.KeyedMessage,
    client = new kafka.KafkaClient({ 
        kafkaHost: '10.129.6.5:9092',
        autoConnect: true
    }),
    producer = new Producer(client),
    Consumer = kafka.Consumer,
    consumer = new Consumer(client,
        [
            { topic: 'response', partition: 0, offset: 0}, 
            { topic: 'heartbeat' }, 
            { topic: "deployed" },
            { topic: "removeWorker" },
            { topic: "RESPONSE_ARBITER_2_DISPATCHER"}
        ],
        [
            { autoCommit: true }
        ])

let db = new Map()
app.use(morgan('combined'))
app.use(bodyParser.urlencoded({ extended: true }))
app.use(bodyParser.json())
const file_path = __dirname + "/repository/"

app.use('/repository', express.static(file_path));
app.use(fileUpload())

let requestQueue = []
let workerNodes = []

const WINDOW_SIZE = 10
const port = 8080
const registry_url = constants.registry_url

/**
 * REST API to receive deployment requests
 */
app.post('/serverless/deploy', (req, res) => {
    
    let runtime = req.body.runtime
    let file = req.files.serverless

    let functionHash = file.md5
    
    file.mv(file_path + functionHash, function (err) {
        functionHash = libSupport.generateExecutor(file_path, functionHash)
        if (err) {
            console.log(err);
            res.send("error").status(400)
        }
        else {
            if (runtime === "container") {
                deployContainer(file_path, functionHash)
                    .then(() => {
                        res.json({
                            status: "success",
                            function_id: functionHash
                        })
                    })
                    .catch(err => {
                        res.json({
                            status: "error",
                            reason: err}).status(400)
                    })
            } else {
                res.json({
                    status: "success",
                    function_id: functionHash
                })
            }
        }
    })
    
})

/**
 * Create the docker file, build and push image to remote repository
 * @param {String: Path from where to extract function executor} path 
 * @param {String: Name of the image} imageName 
 */
function deployContainer(path, imageName) {
    return new Promise((resolve, reject) => {
        let buildStart = Date.now()
        
        fs.writeFile('./repository/Dockerfile',
            `FROM node:latest
            WORKDIR /app
            COPY package.json /app
            RUN npm install
            COPY . /app
            
            CMD node ${imageName}.js`
            , function (err) {
                if (err) {
                    console.log("failed", err);
                    
                    reject(err);
                }
                else {
                    console.log('Dockerfile created');
                    const process = spawn('docker', ["build", "-t", registry_url + imageName, path]);

                    process.stdout.on('data', (data) => {
                        console.log(`stdout: ${data}`);

                    });

                    process.stderr.on('data', (data) => {
                        console.error(`stderr: ${data}`);
                    });

                    process.on('close', (code) => {
                        console.log(`child process exited with code ${code}`);
                        let timeDifference = Math.ceil((Date.now() - buildStart))
                        console.log("image build time taken: ", timeDifference);
                        const process_push = spawn('docker', ["push", registry_url + imageName]);

                        process_push.stdout.on('data', (data) => {
                            console.log(`stdout: ${data}`);

                        });

                        process_push.stderr.on('data', (data) => {
                            console.error(`stderr: ${data}`);
                        });

                        process_push.on('close', (code) => {
                            console.log("image pushed to repository");
                            resolve();
                        })
                        
                    });
                }
            });
    })
}

/**
 * REST API to receive execute requests
 */
app.post('/serverless/execute/:id', (req, res) => {
    let runtime = req.body.runtime
    if (functionToPort.has(req.params.id + runtime)) {
        /**
         * Bypass deployment pipeline if resource available
         */
        let forwardTo = functionToPort.get(req.params.id + runtime) 
        console.log("resource found", forwardTo);
        libSupport.reverseProxy(req, res, `http://${forwardTo.node_id}:${forwardTo.port}/serverless/function/execute`)
        
    } else {
        requestQueue.push({ req, res })
        /**
         * We store functions for function placement heuristics purposes. This lets us look into the function
         * patterns being received and make intelligent deployment decisions based on it.
         */
        if (requestQueue.length >= WINDOW_SIZE)
            dispatch()
    }
})

/**
 * Send dispatch signal and deploy resources after consultation with the RM
 */
function dispatch() {
    let lookbackWindow = Math.min(WINDOW_SIZE, requestQueue.length)
    for (let i = 0; i < lookbackWindow; i++) {
        let {req, res} = requestQueue.shift()
	    console.log(req.body)
        let runtime = req.body.runtime
        let functionHash = req.params.id
        
        let function_id = libSupport.makeid(20) // each function resource request is associated with an unique ID
        console.log("Dispatching function with Id", function_id, runtime);
        let node_id = getAddress() // Requests the RM for address and other metadata for function placement
        
        let payload = [{
            topic: node_id,
            messages: JSON.stringify({
                "type": "execute", // Request sent to Dispatch Daemon via Kafka for actual deployment at the Worker
                function_id,
                runtime, functionHash,
                port: libSupport.getPort(usedPort) // TODO: will be provided by the RM
            }),
            partition: 0
        }]
        rmQueue.set(function_id, payload)
        let payloadToRM = [{
            topic: "REQUEST_DISPATCHER_2_ARBITER",
                messages: JSON.stringify({
                    "id": function_id,
                    "memory": 332,
                }),
                partition: 0
            }]
        producer.send(payloadToRM, () => {
            db.set(functionHash + runtime, { req, res })
        })
        
        
        
    }
}

function getAddress() {
    return workerNodes[Math.floor(Math.random() * workerNodes.length)];
}

app.listen(port, () => console.log(`Server listening on port ${port}!`))

consumer.on('message', function (message) {
    
    let topic = message.topic
        message = message.value
    if (topic === "response") {
        console.log(message);
        
        // message = JSON.parse(message)
        // console.log(message);
        // let {req, res} = db.get(message.function_id)
        // if (res != null)
        //     res.json({
        //         "status": "success",
        //         "reply": message.result
        //     })
        // db.delete(message.function_id)
    } else if (topic === "heartbeat") {
        message = JSON.parse(message)
        if (Date.now() - message.timestamp < 300)
            if (workerNodes.indexOf(message.address) === -1)  {
                workerNodes.push(message.address)
                console.log(workerNodes);
            }
    } else if (topic == "deployed") {
        
        message = JSON.parse(message)
        console.log("deployed", message);
        
        if (db.has(message.functionHash + message.runtime)) {
            let { req, res } = db.get(message.functionHash + message.runtime)
            if (parseInt(message.port) != -1)
                functionToPort.set(message.functionHash + message.runtime, {
                    port: parseInt(message.port),
                    node_id: message.node_id
                })
            
            libSupport.reverseProxy(req, res, 
                `http://${message.node_id}:${message.port}/serverless/function/execute`)
                .then(() => {
                    db.delete(message.functionHash + message.runtime)
                })
        }
    } else if (topic == "removeWorker") {
        console.log("removing metadata", message);
        message = JSON.parse(message)
        usedPort.delete(message.port)
        functionToPort.delete(message.functionHash + message.runtime)
    } else if (topic == "RESPONSE_ARBITER_2_DISPATCHER") {
        
        message = JSON.parse(message)
        console.log(message);
        let payload = rmQueue.get(message.id)
        payload[0].topic = getAddress()
        console.log(payload);
        
        producer.send(payload, () => { })
    }
});

setInterval(dispatch, 2000);
// {
//     id: "!!!!!",
    
//     "grunts": [{
//         id: "a",
//         port: 12121
//      },{
//         id: "b",
//         port: 123445
//      }]
// }