"use strict";

const express = require('express')
const bodyParser = require('body-parser')
const fileUpload = require('express-fileupload');
const constants = require('.././constants.json')
const secrets = require('./secrets.json')
const fs = require('fs')
const { spawn } = require('child_process');
const morgan = require('morgan')
const heap = require('heap')
const fetch = require('node-fetch');

let metadataDB = `http://${secrets.couchdb_username}:${secrets.couchdb_password}@${constants.couchdb_host}`
metadataDB = metadataDB + "/" + constants.couchdb_db_name + "/"

const app = express()
const libSupport = require('./lib')
const logger = libSupport.logger
let date = new Date();
let log_channel = constants.log_channel

let usedPort = new Map(), // TODO: remove after integration with RM
    rmQueue = new Map(), // queue holding requests for which DM is waiting for RM allocation
    db = new Map(), // queue holding request to be dispatched
    resourceMap = new Map(), // map between resource_id and resource details like node_id, port, associated function etc
    functionToResource = new Map() // a function to resource map. Each map contains a minheap of
                                   // resources associated with the function

let kafka = require('kafka-node'),
    Producer = kafka.Producer,
    client = new kafka.KafkaClient({ 
        kafkaHost: constants.kafka_host,
        autoConnect: true
    }),
    producer = new Producer(client),
    Consumer = kafka.Consumer,
    consumer = new Consumer(client,
        [
            { topic: 'heartbeat' }, // receives heartbeat messages from workers, also acts as worker join message
            { topic: "deployed" }, // receives deployment confirmation from workers
            { topic: "removeWorker" }, // received when a executor environment is blown at the worker
            { topic: "RESPONSE_RM_2_DM" }, // receives deployment details from RM
            { topic: "hscale" } // receives signals for horizontal scaling
        ],
        [
            { autoCommit: true }
        ])

app.use(morgan('combined'))
app.use(bodyParser.urlencoded({ extended: true }))
app.use(bodyParser.json())
const file_path = __dirname + "/repository/"

app.use('/repository', express.static(file_path));
app.use(fileUpload())

let requestQueue = []
let workerNodes = []

const WINDOW_SIZE = 10
const port = constants.master_port
const registry_url = constants.registry_url

/**
 * REST API to receive deployment requests
 */
app.post('/serverless/deploy', (req, res) => {
    
    let runtime = req.body.runtime
    let file = req.files.serverless

    let functionHash = file.md5

    file.mv(file_path + functionHash, function (err) {
        functionHash = libSupport.generateExecutor(file_path, functionHash)
        /**
         * Adding meta caching via couchdb
         * This will create / update function related metadata like resource limits etc
         * on a database named "serverless".
         */
        fetch(metadataDB + functionHash).then(res => res.json())
            .then(json => {
                if (json.error === "not_found") {
                    logger.warn("New function, creating metadata")
                    fetch(metadataDB + functionHash, {
                        method: 'put',
                        body: JSON.stringify({
                            memory: req.body.memory
                        }),
                        headers: { 'Content-Type': 'application/json' },
                    }).then(res => res.json())
                    .then(json => console.log(json));
                } else {
                    logger.warn('Repeat deployment, updating metadata')
                    fetch(metadataDB + functionHash, {
                        method: 'put',
                        body: JSON.stringify({
                            memory: req.body.memory,
                            _rev: json._rev
                        }),
                        headers: { 'Content-Type': 'application/json' },
                    }).then(res => res.json())
                    .then(json => console.log(json));
                }
        });
        if (err) {
            logger.error(err)
            res.send("error").status(400)
        }
        else {
            if (runtime === "container") {
                deployContainer(file_path, functionHash)
                    .then(() => {
                        res.json({
                            status: "success",
                            function_id: functionHash
                        })
                    })
                    .catch(err => {
                        res.json({
                            status: "error",
                            reason: err}).status(400)
                    })
            } else {
                res.json({
                    status: "success",
                    function_id: functionHash
                })
            }
        }
    })
    
})

/**
 * Create the docker file, build and push image to remote repository
 * @param {String: Path from where to extract function executor} path 
 * @param {String: Name of the image} imageName 
 */
function deployContainer(path, imageName) {
    return new Promise((resolve, reject) => {
        let buildStart = Date.now()
        
        fs.writeFile('./repository/Dockerfile',
            `FROM node:latest
            WORKDIR /app
            COPY package.json /app
            RUN npm install
            COPY . /app
            ENTRYPOINT ["node", "${imageName}.js"]`
            , function (err) {
                if (err) {
                    logger.error("failed", err);

                    
                    reject(err);
                }
                else {
                    logger.info('Dockerfile created');
                    const process = spawn('docker', ["build", "-t", registry_url + imageName, path]);

                    process.stdout.on('data', (data) => {
                        logger.info(`stdout: ${data}`);

                    });

                    process.stderr.on('data', (data) => {
                        logger.error(`stderr: ${data}`);
                    });

                    process.on('close', (code) => {
                        logger.warn(`child process exited with code ${code}`);
                        let timeDifference = Math.ceil((Date.now() - buildStart))
                        logger.info("image build time taken: ", timeDifference);
                        const process_push = spawn('docker', ["push", registry_url + imageName]);

                        process_push.stdout.on('data', (data) => {
                            console.log(`stdout: ${data}`);

                        });

                        process_push.stderr.on('data', (data) => {
                            logger.error(`stderr: ${data}`);
                        });

                        process_push.on('close', (code) => {
                            logger.info("image pushed to repository");
                            resolve();
                        })
                        
                    });
                }
            });
    })
}

/**
 * REST API to receive execute requests
 */
app.post('/serverless/execute/:id', (req, res) => {
    let runtime = req.body.runtime
    let id = req.params.id + runtime
    if (functionToResource.has(id)) {
        libSupport.reverseProxy(req, res, functionToResource, resourceMap)
    } else {
        /**
         * Requests are queued up before being dispatched. To prevent requests coming in for the 
         * same function from starting too many workers, they are grouped together
         * and one worker is started per group.
         */
        if (db.has(req.params.id + runtime)) {
            db.get(req.params.id + runtime).push({ req, res })
            return;
        }
        requestQueue.push({ req, res })
        /**
         * We store functions for function placement heuristics purposes. This lets us look into the function
         * patterns being received and make intelligent deployment decisions based on it.
         */
        if (requestQueue.length >= WINDOW_SIZE)
            dispatch()
    }
})

/**
 * Send dispatch signal to Worker nodes and deploy resources after consultation with the RM
 */
function dispatch() {
    let lookbackWindow = Math.min(WINDOW_SIZE, requestQueue.length)
    for (let i = 0; i < lookbackWindow; i++) {
        let {req, res} = requestQueue.shift()
        
	    // logger.info(req.body)
        let runtime = req.body.runtime
        let functionHash = req.params.id
        if (!db.has(functionHash + runtime)) {
            db.set(functionHash + runtime, [])
            db.get(functionHash + runtime).push({ req, res })
            let resource_id = libSupport.makeid(20) // each function resource request is associated with an unique ID
            logger.info(`Generated new resource ID: ${resource_id} for runtime: ${runtime}`);
            
            let node_id = getAddress() // Requests the RM for address and other metadata for function placement
            let port = libSupport.getPort(usedPort) // TODO: will be provided by the RM

            let payload = [{
                topic: node_id,
                messages: JSON.stringify({
                    "type": "execute", // Request sent to Dispatch Daemon via Kafka for actual deployment at the Worker
                    resource_id,
                    runtime, functionHash,
                    port
                }),
                partition: 0
            }]
            logger.info("Requesting RM " + JSON.stringify({
                resource_id,
                "memory": 332,
            }))

            /** uncomment when RM is unavailable */
            
            resourceMap.set(resource_id, {
                runtime, functionHash, port, node_id
            })
            logger.info(resourceMap);
            producer.send(payload, () => {
                logger.info(`Resource Deployment request sent to Dispatch Agent`)
            })
        } else {
            logger.info("deployment process already started waiting")
            db.get(functionHash + runtime).push({ req, res })
        }
        

        
        /** uncomment when RM is available, TODO: also update resourceMap
        rmQueue.set(resource_id, payload)
        let payloadToRM = [{
            topic: "REQUEST_DM_2_RM",
                messages: JSON.stringify({
                    resource_id,
                    "memory": 332,
                }),
                partition: 0
            }]
        producer.send(payloadToRM, () => {
            db.set(functionHash + runtime, { req, res })
        })
         */
        
    }
}

function getAddress() {
    return workerNodes[Math.floor(Math.random() * workerNodes.length)];
}

function postDeploy(message) {
    logger.info("Deployed Resource: " + JSON.stringify(message));

    if (message.status == false) {
        let sendQueue = db.get(message.functionHash + message.runtime)
        // TODO: handle failure
        while (sendQueue && sendQueue.length != 0) {
            let { req, res } = sendQueue.shift()
            res.status(400).json({ reason: message.reason })
        }
        db.delete(message.functionHash + message.runtime)
        
        return;
    }
    if (functionToResource.has(message.functionHash + message.runtime)) {
        let resourceHeap = functionToResource.get(message.functionHash + message.runtime)
        heap.push(resourceHeap, {
            resource_id: message.resource_id,
            metric: 0
        }, libSupport.compare)
        logger.warn("Horizontally scaling up: " +
            JSON.stringify(functionToResource.get(message.functionHash + message.runtime)));

    } else {
        /**
        * function to resource map - holds a min heap of resources associated with a function
        * the min heap is sorted based on a metric [TBD] like CPU usage, request count, mem usage etc
        * TODO: decide on metric to use for sorting.
        */
        let resourceHeap = []
        heap.push(resourceHeap, {
            resource_id: message.resource_id,
            metric: 0
        }, libSupport.compare)
        functionToResource.set(message.functionHash + message.runtime, resourceHeap)
        logger.warn("Creating new resource pool"
            + JSON.stringify(functionToResource.get(message.functionHash + message.runtime)));

    }
    let resource = resourceMap.get(message.resource_id)
    let confirmRM = [{
        topic: log_channel,
        messages: JSON.stringify({
            resource_id: message.resource_id,
            node_id: resource.node_id,
            runtime: resource.runtime,
            function_id: resource.functionHash,
            "reason": "deployment",
            "status": true,
            "timestamp": date.toISOString()
        }),
        partition: 0
    }]
    producer.send(confirmRM, () => {
        logger.info(`Confirmed RM for successful deployment resource_id: ${message.resource_id} deployment`)
    })

    if (db.has(message.functionHash + message.runtime)) {
        let sendQueue = db.get(message.functionHash + message.runtime)
        logger.info("forwarding request via reverse proxy to: " + JSON.stringify(resource));
        while (sendQueue && sendQueue.length != 0) {
            let { req, res } = sendQueue.shift()
            libSupport.reverseProxy(req, res, functionToResource, resourceMap)
                .then(() => {

                })
        }
        db.delete(message.functionHash + message.runtime)
    }

}

consumer.on('message', function (message) {
    
    let topic = message.topic
        message = message.value
    if (topic === "response") {
        logger.info("response " + message);
        
        
    } else if (topic === "heartbeat") {
        message = JSON.parse(message)
        if (Date.now() - message.timestamp < 1000)
            if (workerNodes.indexOf(message.address) === -1)  {
                workerNodes.push(message.address)
                logger.warn("New worker discovered. Worker List: " + workerNodes)
            }
    } else if (topic == "deployed") {
        try {
            message = JSON.parse(message)
        } catch (e) {
            // process.exit(0)
        }
        postDeploy(message)
        
    } else if (topic == "removeWorker") {
        logger.warn("Worker blown: Removing Metadata " + message);
        try {
            message = JSON.parse(message)
        } catch(e) {
            // process.exit(0)
        }
        usedPort.delete(message.port)
        if (functionToResource.has(message.functionHash + message.runtime)) {
            let resourceArray = functionToResource.get(message.functionHash + message.runtime)
            for (let i = 0; i < resourceArray.length; i++)
                if (resourceArray[i].resource_id === message.resource_id) {
                    resourceArray.splice(i, 1);
                    break;
                }

            heap.heapify(resourceArray, libSupport.compare)
            
            resourceMap.delete(message.resource_id)
            if (resourceArray.length == 0)
                functionToResource.delete(message.functionHash + message.runtime)
            
        }
        

    } else if (topic == "hscale") {
        message = JSON.parse(message)
        let resource_id = libSupport.makeid(20), // each function resource request is associated with an unique ID
            node_id = getAddress(), // Requests the RM for address and other metadata for function placement
            port = libSupport.getPort(usedPort), // TODO: will be provided by the RM
            runtime = message.runtime,
            functionHash = message.functionHash

        let payload = [{
            topic: node_id,
            messages: JSON.stringify({
                "type": "execute", // Request sent to Dispatch Daemon via Kafka for actual deployment at the Worker
                resource_id,
                runtime, functionHash,
                port
            }),
            partition: 0
        }]
        logger.info("Requesting RM " + JSON.stringify({
            resource_id,
            "memory": 332,
        }))

        /** uncomment when RM is unavailable */

        resourceMap.set(resource_id, {
            runtime, functionHash, port, node_id
        })
        logger.info(resourceMap);
        producer.send(payload, () => {
            logger.info(`Resource Deployment request sent to Dispatch Agent`)
        })
    } else if (topic == "RESPONSE_RM_2_DM") {
        
        logger.info("Response from RM: " + message);
        message = JSON.parse(message)
        
        let payload = rmQueue.get(message.id)
        if (payload != null) {
            payload[0].topic = message.nodes[0]
            logger.info(payload);
            /** get port and other resources */
            let resource = resourceMap.get(message.id)
            resource.node_id = message.nodes[0] // TODO: update this to message.nodes[0].node_id
            // resource.port = message.nodes[0].port TODO: update after RM supports port allocation
            resourceMap.set(message.id, resource)
            logger.info(resourceMap);
            producer.send(payload, () => { })
        } else {
            logger.error("something went wrong");
            
        }
        
    }
});

function autoscalar() {
    functionToResource.forEach((resourceList, functionKey, map) => {
        console.log(resourceList);
        
        if (resourceList.length > 0 && resourceList[resourceList.length - 1].metric > 100) {
            let resource = resourceMap.get(resourceList[resourceList.length - 1].resource_id)
            logger.warn(`resource ${resourceList[resourceList.length - 1]} exceeded autoscalar threshold. Scaling up!`)
            let payload = [{
                topic: "hscale",
                messages: JSON.stringify({ "runtime": resource.runtime, "functionHash": resource.functionHash })
            }]
            producer.send(payload, function () { })
        }
    });

}


setInterval(autoscalar, 1000);
setInterval(dispatch, 1000);
app.listen(port, () => logger.info(`Server listening on port ${port}!`))