package hpdos.lib;

import hpdos.message.MessageConstants;

import java.util.concurrent.ConcurrentHashMap;

public class MemoryStorageService implements StorageService {
    private final ConcurrentHashMap<String, StorageModel> memoryKVStore;
    public MemoryStorageService() {
        this.memoryKVStore = new ConcurrentHashMap<>();
    }
    @Override
    public StoredModel create(String key, StorageModel value) {
        try {
            if (memoryKVStore.putIfAbsent(key, value) != null)
                return new StoredModel(null, MessageConstants.STATUS_KEY_EXISTS);
            return new StoredModel(value, MessageConstants.STATUS_OK);
        } catch (Exception e) {
            return null;
        }
    }

    @Override
    public StoredModel readByKey(String key) {
        if (!memoryKVStore.containsKey(key))
            return new StoredModel(null, MessageConstants.STATUS_KEY_NOT_FOUND);
        return new StoredModel(memoryKVStore.get(key), MessageConstants.STATUS_OK);
    }

    @Override
    public StoredModel update(String key, StorageModel value) {
        StorageModel previousValue = memoryKVStore.get(key);
        if (previousValue == null)
            return new StoredModel(null, MessageConstants.STATUS_KEY_NOT_FOUND);
        else if (previousValue.getAccessType() == MessageConstants.METADATA_ACCESS_PRIVATE
                && !previousValue.getOwner().equals(value.getOwner()))
            return new StoredModel(null, MessageConstants.STATUS_UNAUTHORIZED_PRIVATE_KEY_ACCESS);

        // the request will have the old version number of the data to be inserted, we only update the data
        // with a new version number if at the time of update the two versions match
        // else we reject the update
        StorageModel newValue = value.createVersionUpdatedModel();
        boolean status = memoryKVStore.replace(key, value, newValue); // the equals method is overridden in Storage model
                                                  // to equate two objects based on their version numbers
        if (status)
            return new StoredModel(value, MessageConstants.STATUS_OK);
        else
            return new StoredModel(null, MessageConstants.STATUS_UPDATE_VERSION_MISMATCH);
    }

    @Override
    public StoredModel delete(String key, StorageModel value) {
        StorageModel previousValue = memoryKVStore.get(key);
        if (previousValue == null)
            return new StoredModel(null, MessageConstants.STATUS_KEY_NOT_FOUND);
        else if (previousValue.getAccessType() == MessageConstants.METADATA_ACCESS_PRIVATE
                && !previousValue.getOwner().equals(value.getOwner()))
            return new StoredModel(null, MessageConstants.STATUS_UNAUTHORIZED_PRIVATE_KEY_ACCESS);
        boolean status = memoryKVStore.remove(key, value);
        if (status)
            return new StoredModel(previousValue, MessageConstants.STATUS_OK);
        else
            return new StoredModel(null, MessageConstants.STATUS_UPDATE_VERSION_MISMATCH);
    }
}
