package hpdos.lib;

import com.google.common.base.Stopwatch;
import com.google.common.util.concurrent.ListenableFuture;
import hpdos.ConfigConstants;
import hpdos.grpc.ReplicationRequest;
import hpdos.grpc.ReplicationResponse;
import hpdos.grpc.ReplicationServiceGrpc;
import hpdos.grpc.Response;
import hpdos.message.MessageConstants;
import hpdos.message.ResponseBuilder;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;

import java.util.*;
import java.util.concurrent.*;

public class InlineReplicationService implements ReplicationService {

    private final HashMap<String, MasterFollower> followers;
    private final HashMap<String, ManagedChannel> channels;
    private final ExecutorService executorService;
    public InlineReplicationService(HashMap<String, MasterFollower> followers) {
        this.followers = followers;
        this.channels = new HashMap<>();
        for (MasterFollower follower: this.followers.values()) {
            ManagedChannel channel = ManagedChannelBuilder
                    .forAddress(follower.getIp(), follower.getPort())
                    .usePlaintext()
                    .build();
            channels.put(follower.getFollowerID(), channel);
        }
        this.executorService = Executors.newFixedThreadPool(ConfigConstants.REPLICATOR_THREAD_POOL_SIZE);
    }

    @Override
    public void cleanup() throws InterruptedException {
        for (ManagedChannel channel: channels.values())
            channel.shutdown();
        executorService.shutdown();
        executorService.awaitTermination(MessageConstants.STATUS_REPLICATION_TIMEOUT, TimeUnit.MILLISECONDS);
    }

    private void establishChannels() {
        for (String followerID: followers.keySet()) {
            if (!channels.containsKey(followerID)) {
                MasterFollower follower = followers.get(followerID);
                ManagedChannel channel = ManagedChannelBuilder
                        .forAddress(follower.getIp(), follower.getPort())
                        .usePlaintext()
                        .build();
                channels.put(follower.getFollowerID(), channel);
            }
        }
    }

    @Override
    public ReplicationResponse replicateMetadata(ReplicationRequest replicationRequest)
            throws InterruptedException, ExecutionException {

        Set<Callable<ReplicationResponse>> callables = new HashSet<>();
        // new followers have joined or left.
        // TODO: Handle follower leaving scenario
        // FIXME: fix edge case where equal number of followers leaving and joining won't trigger connection reestablishment
        if (channels.size() != followers.size()) {
            establishChannels();
        }
        for (ManagedChannel channel: channels.values()) {
            callables.add(() -> {
                ReplicationServiceGrpc.ReplicationServiceBlockingStub stub =
                        ReplicationServiceGrpc.newBlockingStub(channel);
                return stub.replicateMetadata(replicationRequest);
            });
        }
        List<Future<ReplicationResponse>> futures = executorService.invokeAll(callables);
        HashMap<String, Response> responseHashMap = new HashMap<>();
        Stopwatch stopwatch = Stopwatch.createUnstarted();
        stopwatch.start();
        for (Future<ReplicationResponse> future: futures) {
            ReplicationResponse replicationResponse;
            replicationResponse = future.get(); //TODO: Add and handle get timeout. Timeout related constants already added

            for (Response receivedResponse: replicationResponse.getResponseList()) {
                int status = receivedResponse.getStatus();
                if (status == MessageConstants.STATUS_OK) {
                    if (!responseHashMap.containsKey(receivedResponse.getAck().getKey()))
                        responseHashMap.put(receivedResponse.getAck().getKey(), receivedResponse);
                } else {
                    responseHashMap.put(receivedResponse.getNack().getKey(), receivedResponse);
                }
            }
        }
        stopwatch.stop();
//        System.out.println("replicateMetadata ReplicationService " + stopwatch);
        return ResponseBuilder.
                buildReplicationResponse(new ArrayList<>(responseHashMap.values()));
    }

    @Override
    public ReplicationResponse replicateMetadataAsync(ReplicationRequest replicationRequest) {
        for (ManagedChannel channel: channels.values()) {
            ReplicationServiceGrpc.ReplicationServiceFutureStub stub =
                    ReplicationServiceGrpc.newFutureStub(channel);
            ListenableFuture<ReplicationResponse> res = stub.replicateMetadata(replicationRequest);

        }
        throw new UnsupportedOperationException("Implementation not complete");
    }

    @Override
    public HashMap<String, MasterFollower> getFollowers() {
        return followers;
    }
}
