package hpdos.lib;

import com.google.common.base.Stopwatch;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import hpdos.ConfigConstants;
import hpdos.grpc.ReplicationRequest;
import hpdos.grpc.ReplicationResponse;
import hpdos.grpc.ReplicationServiceGrpc;
import hpdos.grpc.Response;
import hpdos.message.MessageConstants;
import hpdos.message.ResponseBuilder;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;

import javax.annotation.Nonnull;
import java.util.*;
import java.util.concurrent.*;

public class InlineReplicationService implements ReplicationService {

    private final HashMap<String, MasterFollower> followers;
    private final HashMap<String, ManagedChannel> channels;
    private ExecutorService executorService;
    private final boolean isReplicationAsync;
    public InlineReplicationService(HashMap<String, MasterFollower> followers, Properties properties) {
        this.followers = followers;
        this.channels = new HashMap<>();
        String replicationType = (String) properties.get("app.REPLICATION_TYPE");
        this.isReplicationAsync = replicationType.equals(ConfigConstants.replicationAsync);
        if (!this.isReplicationAsync) {
            int replicationThreadPoolSize;
            if (properties.containsKey("app.REPLICATOR_THREAD_POOL_SIZE")) {
                replicationThreadPoolSize = Integer.parseInt((String)
                        properties.get("app.REPLICATOR_THREAD_POOL_SIZE"));
            } else {
                replicationThreadPoolSize = ConfigConstants.REPLICATOR_THREAD_POOL_SIZE;
            }
            this.executorService = Executors.newFixedThreadPool(replicationThreadPoolSize);
            System.out.println("Creating synchronous replication pool. Pool size: " + replicationThreadPoolSize);
        } else {
            System.out.println("Replication to be handled using asynchronous handlers");
        }
    }

    @Override
    public void cleanup() throws InterruptedException {
        for (ManagedChannel channel: channels.values())
            channel.shutdown();
        if (this.executorService != null) {
            executorService.shutdown();
            boolean status = executorService.
                    awaitTermination(MessageConstants.STATUS_REPLICATION_TIMEOUT, TimeUnit.MILLISECONDS);
            if (!status)
                executorService.shutdownNow();

        }
    }

    private void establishChannels() {
        for (String followerID: followers.keySet()) {
            if (!channels.containsKey(followerID)) {
                MasterFollower follower = followers.get(followerID);
                ManagedChannel channel = ManagedChannelBuilder
                        .forAddress(follower.getIp(), follower.getPort())
                        .usePlaintext()
                        .build();
                channels.put(follower.getFollowerID(), channel);
            }
        }
    }

    @Override
    public ReplicationResponse replicateMetadata(ReplicationRequest replicationRequest) throws ExecutionException, InterruptedException {
        if (this.isReplicationAsync) {
            return replicateMetadataAsync(replicationRequest);
        } else {
            return replicateMetadataSync(replicationRequest);
        }
    }
    public ReplicationResponse replicateMetadataSync(ReplicationRequest replicationRequest)
            throws InterruptedException, ExecutionException {

        Set<Callable<ReplicationResponse>> callables = new HashSet<>();
        // new followers have joined or left.
        // TODO: Handle follower leaving scenario
        // FIXME: fix edge case where equal number of followers leaving and joining won't trigger connection reestablishment
        if (channels.size() != followers.size()) {
            establishChannels();
        }
        for (ManagedChannel channel: channels.values()) {
            callables.add(() -> {
                ReplicationServiceGrpc.ReplicationServiceBlockingStub stub =
                        ReplicationServiceGrpc.newBlockingStub(channel);
                return stub.replicateMetadata(replicationRequest);
            });
        }
        List<Future<ReplicationResponse>> futures = executorService.invokeAll(callables);
        HashMap<String, Response> responseHashMap = new HashMap<>();
        Stopwatch stopwatch = Stopwatch.createUnstarted();
        stopwatch.start();
        for (Future<ReplicationResponse> future: futures) {
            ReplicationResponse replicationResponse;
            replicationResponse = future.get(); //TODO: Add and handle get timeout. Timeout related constants already added

            for (Response receivedResponse: replicationResponse.getResponseList()) {
                int status = receivedResponse.getStatus();
                if (status == MessageConstants.STATUS_OK) {
                    if (!responseHashMap.containsKey(receivedResponse.getAck().getKey()))
                        responseHashMap.put(receivedResponse.getAck().getKey(), receivedResponse);
                } else {
                    responseHashMap.put(receivedResponse.getNack().getKey(), receivedResponse);
                }
            }
        }
        stopwatch.stop();
//        System.out.println("replicateMetadata ReplicationService " + stopwatch);
        return ResponseBuilder.
                buildReplicationResponse(new ArrayList<>(responseHashMap.values()));
    }

    /**
     * Incomplete implementation do not use
     * @param replicationRequest replication request sent
     * @return replication response
     */
    public ReplicationResponse replicateMetadataAsync(ReplicationRequest replicationRequest) throws InterruptedException {
        CountDownLatch replicationWaiter = new CountDownLatch(this.followers.size());
        HashMap<String, Response> responseHashMap = new HashMap<>();
        if (channels.size() != followers.size()) {
            establishChannels();
        }
        for (ManagedChannel channel: channels.values()) {
            ReplicationServiceGrpc.ReplicationServiceFutureStub stub =
                    ReplicationServiceGrpc.newFutureStub(channel);
            ListenableFuture<ReplicationResponse> res = stub.replicateMetadata(replicationRequest);
            Futures.addCallback(res, new FutureCallback<>() {
                @Override
                public void onSuccess(ReplicationResponse result) {

                    for (Response receivedResponse: result.getResponseList()) {
                        int status = receivedResponse.getStatus();
                        if (status == MessageConstants.STATUS_OK) {
                            if (!responseHashMap.containsKey(receivedResponse.getAck().getKey()))
                                responseHashMap.put(receivedResponse.getAck().getKey(), receivedResponse);
                        } else {
                            responseHashMap.put(receivedResponse.getNack().getKey(), receivedResponse);
                        }
                    }

                    replicationWaiter.countDown();
                }

                @Override
                public void onFailure(@Nonnull Throwable t) {
                    replicationWaiter.countDown();
                }
            }, MoreExecutors.directExecutor());
        }

        Stopwatch stopwatch = Stopwatch.createUnstarted();
        stopwatch.start();
        replicationWaiter.await();
        stopwatch.stop();
//        System.out.println("replication latency" + stopwatch);
        return ResponseBuilder.
                buildReplicationResponse(new ArrayList<>(responseHashMap.values()));
    }

    @Override
    public HashMap<String, MasterFollower> getFollowers() {
        return followers;
    }
}
