package hpdos.handler;

import com.google.common.base.Stopwatch;
import hpdos.grpc.*;
import hpdos.lib.MasterFollower;
import hpdos.lib.ReplicationService;
import hpdos.message.RequestBuilder;
import hpdos.message.ResponseBuilder;
import io.grpc.stub.StreamObserver;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.concurrent.ExecutionException;

public class NetworkHandler extends NetworkServiceGrpc.NetworkServiceImplBase {
    private final IOHandler ioHandler;
    private final ReplicationService replicationService;
    public NetworkHandler(IOHandler ioHandler, ReplicationService replicationService) {
        this.ioHandler = ioHandler;
        this.replicationService = replicationService;
    }

    @Override
    public void readMetadata(Packet requestPacket,
                        StreamObserver<Packet> responseObserver) {
//        System.out.println("Data received" + requestPacket.toString());
        for (Request request: requestPacket.getRequestList()) {
            Response response = ioHandler.read(request);
            Packet packet = ResponseBuilder.buildPacket(response);
//            System.out.println(packet);
            responseObserver.onNext(packet);
        }

        responseObserver.onCompleted();
    }

    @Override
    public void createMetadata(Packet requestPacket, StreamObserver<Packet> responseObserver) {
        Stopwatch stopwatch = Stopwatch.createUnstarted();
        stopwatch.start();
        for (Request request: requestPacket.getRequestList()) {
            ioHandler.create(request);
        }
        stopwatch.stop();

//        System.out.println("Added to local memory " + stopwatch);
        stopwatch.reset();
        stopwatch.start();
        Packet packet = replicate(requestPacket);
        stopwatch.stop();
//        System.out.println("Replication time " + stopwatch);
        responseObserver.onNext(packet);
        responseObserver.onCompleted();
    }

    @Override
    public void updateMetadata(Packet requestPacket, StreamObserver<Packet> responseObserver) {
//        System.out.println("new create request " + requestPacket);
        for (Request request: requestPacket.getRequestList()) {
            ioHandler.update(request);
        }
//        System.out.println("Added to local memory");
        Packet packet = replicate(requestPacket);
        responseObserver.onNext(packet);
        responseObserver.onCompleted();
    }

    @Override
    public void deleteMetadata(Packet requestPacket, StreamObserver<Packet> responseObserver) {
//        System.out.println("new create request " + requestPacket);
        for (Request request: requestPacket.getRequestList()) {
            ioHandler.delete(request);
        }
//        System.out.println("Added to local memory");
        Packet packet = replicate(requestPacket);
        responseObserver.onNext(packet);
        responseObserver.onCompleted();
    }

    @Override
    public void getReadReplicaList(RequestList requestList,
                                   StreamObserver<ResponseList> responseListStreamObserver) {
        HashMap<String, MasterFollower> followers = replicationService.getFollowers();
        ResponseList.Builder responseList = ResponseList.newBuilder();
        for (MasterFollower follower: followers.values()) {
            Follower.Builder replica = Follower.newBuilder();
            replica.setIp(follower.getIp());
            replica.setPort(follower.getPort());
            replica.setFollowerID(follower.getFollowerID());
            replica.setLastSeen(follower.getLastSeen());
            replica.build();
            responseList.addFollower(replica);
        }
        responseListStreamObserver.onNext(responseList.build());
        responseListStreamObserver.onCompleted();
    }

    private Packet replicate(Packet requestPacket) {
        ReplicationRequest replicationRequest = RequestBuilder.
                buildReplicationRequest(new ArrayList<>(requestPacket.getRequestList()));
        ReplicationResponse replicationResponse = null;
        try {
//            System.out.println("starting replication");
            Stopwatch stopwatch = Stopwatch.createUnstarted();
            stopwatch.start();
            replicationResponse = replicationService.replicateMetadata(replicationRequest);
            stopwatch.stop();
//            System.out.println("Network handler replicate" + stopwatch);
        } catch (InterruptedException | ExecutionException e) {
            e.printStackTrace();
        }
//        System.out.println("replication complete");
        Packet packet = ResponseBuilder.buildPacket(new ArrayList<>(replicationResponse.getResponseList()));
//        System.out.println(packet);
        return packet;
    }
}
