package hpdos.handler;

import hpdos.grpc.Ack;
import hpdos.grpc.Nack;
import hpdos.grpc.Request;
import hpdos.grpc.Response;
import hpdos.lib.StorageModel;
import hpdos.lib.StorageService;
import hpdos.lib.StoredModel;
import hpdos.message.MessageConstants;
import hpdos.message.ResponseBuilder;

public class IOHandler {
    StorageService storageService;
    private final boolean isMaster;
    public IOHandler(StorageService storageService, boolean isMaster) {
        this.storageService = storageService;
        this.isMaster = isMaster;
    }

    public Response create(Request request) {
        Ack ack = null; Nack nack = null;
        StorageModel blob = new StorageModel(
                MessageConstants.INIT_VERSION,
                request.getDataSize(),
                request.getKey(),
                request.getAccessType(),
                request.getClientID(),
                request.getValue());
        StoredModel storedData = storageService.create(request.getKey(), blob);
        if (storedData.getStatus() == MessageConstants.STATUS_OK) {
            ack = ResponseBuilder.buildAck(storedData.getData().getVersion(),
                    storedData.getData().getDataSize(),
                    storedData.getData().getKey(),
                    storedData.getData().getCrc(),
                    storedData.getData().getValue());

            return ResponseBuilder.buildResponsePacket(
                    MessageConstants.METADATA_CREATE, MessageConstants.STATUS_OK,
                    ack, null);
        }
        else {
            nack = ResponseBuilder.buildNack(MessageConstants.INVALID_VERSION,
                    request.getKey());
            return ResponseBuilder.buildResponsePacket(
                    MessageConstants.METADATA_CREATE, storedData.getStatus(),
                    null, nack);
        }
    }

    public Response update(Request request) {
        Ack ack = null; Nack nack = null;
        StorageModel blob = new StorageModel(
                request.getVersion(),
                request.getDataSize(),
                request.getKey(),
                request.getAccessType(),
                request.getClientID(),
                request.getValue());
        StoredModel storedData = storageService.update(request.getKey(), blob);
        if (storedData.getStatus() == MessageConstants.STATUS_OK) {
            ack = ResponseBuilder.buildAck(storedData.getData().getVersion(),
                    storedData.getData().getDataSize(),
                    storedData.getData().getKey(),
                    storedData.getData().getCrc(),
                    storedData.getData().getValue());

            return ResponseBuilder.buildResponsePacket(
                    MessageConstants.METADATA_UPDATE, MessageConstants.STATUS_OK,
                    ack, null);
        }
        else {
            int version = (storedData.getStatus() == MessageConstants.STATUS_UPDATE_VERSION_MISMATCH)?
                    storedData.getData().getVersion(): MessageConstants.INVALID_VERSION;
            nack = ResponseBuilder.buildNack(version, request.getKey());
            return ResponseBuilder.buildResponsePacket(
                    MessageConstants.METADATA_UPDATE, storedData.getStatus(),
                    null, nack);
        }
    }

    /**
     * Removes the object from the KV store if the request version matches with the current version of data
     * @param request Models to be deleted. Only the key, clientID and version needs to be valid
     * @return previous value if successful else null with status
     */
    public Response delete(Request request) {
        Ack ack = null; Nack nack = null;
        StorageModel blob = new StorageModel(
                request.getVersion(),
                request.getDataSize(),
                request.getKey(),
                request.getAccessType(),
                request.getClientID(),
                request.getValue());
        StoredModel storedData = storageService.delete(request.getKey(), blob);
        if (storedData.getStatus() == MessageConstants.STATUS_OK) {
            ack = ResponseBuilder.buildAck(storedData.getData().getVersion(),
                    storedData.getData().getDataSize(),
                    storedData.getData().getKey(),
                    storedData.getData().getCrc(),
                    storedData.getData().getValue());

            return ResponseBuilder.buildResponsePacket(
                    MessageConstants.METADATA_DELETE, MessageConstants.STATUS_OK,
                    ack, null);
        }
        else {
            int version = (storedData.getStatus() == MessageConstants.STATUS_UPDATE_VERSION_MISMATCH)?
                    storedData.getData().getVersion(): MessageConstants.INVALID_VERSION;
            nack = ResponseBuilder.buildNack(version, request.getKey());
            return ResponseBuilder.buildResponsePacket(
                    MessageConstants.METADATA_DELETE, storedData.getStatus(),
                    null, nack);
        }
    }

    public Response read(Request request) {
        Ack ack = null; Nack nack = null;
        int status;
        // Only Metadata Master can serve shared object lookup requests
        if (!isMaster && request.getAccessType() == MessageConstants.METADATA_ACCESS_SHARED) {
            status = MessageConstants.STATUS_SERVER_NOT_MASTER;
            nack = ResponseBuilder.buildNack(MessageConstants.INVALID_VERSION,
                    request.getKey());
        } else {
            StoredModel readData = storageService.readByKey(request.getKey());
            if (readData.getStatus() == MessageConstants.STATUS_KEY_NOT_FOUND) {
                status = MessageConstants.STATUS_KEY_NOT_FOUND;
                nack = ResponseBuilder.buildNack(MessageConstants.INVALID_VERSION,
                        request.getKey());

            } else if (readData.getData() != null &&
                    readData.getData().getAccessType() == MessageConstants.METADATA_ACCESS_PRIVATE &&
                    !request.getClientID().equals(readData.getData().getOwner())) {
                status = MessageConstants.STATUS_UNAUTHORIZED_PRIVATE_KEY_ACCESS;
                nack = ResponseBuilder.buildNack(MessageConstants.INVALID_VERSION,
                        request.getKey());
            } else {
                status = MessageConstants.STATUS_OK;
                ack = ResponseBuilder.buildAck(readData.getData().getVersion(),
                        readData.getData().getDataSize(),
                        readData.getData().getKey(),
                        readData.getData().getCrc(),
                        readData.getData().getValue());
            }
        }
        return ResponseBuilder.buildResponsePacket(
                MessageConstants.METADATA_READ, status,
                ack, nack);
    }
}
