package com.sudogeeks.talking_reminder;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;

import java.util.ArrayList;
import java.util.List;

/**\brief Class to manage database
 * This class is for managing the database
 */
public class ReminderDatabase extends SQLiteOpenHelper {
    // Database Version
    private static final int DATABASE_VERSION = 1;

    // Database Name
    private static final String DATABASE_NAME = "ReminderDatabase";

    // Table name
    private static final String TABLE_REMINDERS = "ReminderTable";

    // Table Columns names
    private static final String KEY_ID = "id";
    private static final String KEY_TITLE = "title";
    private static final String KEY_DATE = "date";
    private static final String KEY_TIME = "time";
    private static final String KEY_REPEAT = "repeat";
    private static final String KEY_REPEAT_NO = "repeat_no";
    private static final String KEY_REPEAT_TYPE = "repeat_type";
    private static final String KEY_ACTIVE = "active";

    public ReminderDatabase(Context context) {
        super(context, DATABASE_NAME, null, DATABASE_VERSION);
    }

    /**
     * Creating Tables for reminder
     * @param db
     */

    @Override
    public void onCreate(SQLiteDatabase db) {
        String CREATE_REMINDERS_TABLE = "CREATE TABLE " + TABLE_REMINDERS +
                "("
                + KEY_ID + " INTEGER PRIMARY KEY,"
                + KEY_TITLE + " TEXT,"
                + KEY_DATE + " TEXT,"
                + KEY_TIME + " INTEGER,"
                + KEY_REPEAT + " BOOLEAN,"
                + KEY_REPEAT_NO + " INTEGER,"
                + KEY_REPEAT_TYPE + " TEXT,"
                + KEY_ACTIVE + " BOOLEAN" + ")";
        db.execSQL(CREATE_REMINDERS_TABLE);
    }

    /**
     * Upgrading database
     * @param db
     * @param oldVersion
     * @param newVersion
     */

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        // Drop older table if existed
        if (oldVersion >= newVersion)
            return;
        db.execSQL("DROP TABLE IF EXISTS " + TABLE_REMINDERS);

        // Create tables again
        onCreate(db);
    }

    /**
     * This method is for adding ReminderDO into the reminder table in database
     * @param reminderDO
     * @return
     */

    public int addReminder(ReminderDO reminderDO) {
        SQLiteDatabase db = this.getWritableDatabase();
        ContentValues values = new ContentValues();

        values.put(KEY_TITLE, reminderDO.getTitle());
        values.put(KEY_DATE, reminderDO.getDate());
        values.put(KEY_TIME, reminderDO.getTime());
        values.put(KEY_REPEAT, reminderDO.getRepeat());
        values.put(KEY_REPEAT_NO, reminderDO.getRepeatNo());
        values.put(KEY_REPEAT_TYPE, reminderDO.getRepeatType());
        values.put(KEY_ACTIVE, reminderDO.getActive());

        // Inserting Row
        long ID = db.insert(TABLE_REMINDERS, null, values);
        db.close();
        return (int) ID;
    }

    /**
     * To get single ReminderDO corresponding to the id in parameter
     * @param id
     * @return reminderDo
     */
    public ReminderDO getReminder(int id) {
        SQLiteDatabase db = this.getReadableDatabase();

        Cursor cursor = db.query(TABLE_REMINDERS, new String[]
                        {
                                KEY_ID,
                                KEY_TITLE,
                                KEY_DATE,
                                KEY_TIME,
                                KEY_REPEAT,
                                KEY_REPEAT_NO,
                                KEY_REPEAT_TYPE,
                                KEY_ACTIVE
                        }, KEY_ID + "=?",

                new String[]{String.valueOf(id)}, null, null, null, null);

        if (cursor != null)
            cursor.moveToFirst();

        ReminderDO reminderDO = new ReminderDO(Integer.parseInt(cursor.getString(0)), cursor.getString(1),
                cursor.getString(2), cursor.getString(3), cursor.getString(4),
                cursor.getString(5), cursor.getString(6), cursor.getString(7));

        return reminderDO;
    }

    /**
     * This method is for getting all reminders for a particular user
     * @return List of <ReminderDo>
     */
    // Getting all Reminders
    public List<ReminderDO> getAllReminders() {
        List<ReminderDO> reminderDOList = new ArrayList<>();

        // Select all Query
        String selectQuery = "SELECT * FROM " + TABLE_REMINDERS;

        SQLiteDatabase db = this.getWritableDatabase();
        Cursor cursor = db.rawQuery(selectQuery, null);

        // Looping through all rows and adding to list
        if (cursor.moveToFirst()) {
            do {
                ReminderDO reminderDO = new ReminderDO();
                reminderDO.setID(Integer.parseInt(cursor.getString(0)));
                reminderDO.setTitle(cursor.getString(1));
                reminderDO.setDate(cursor.getString(2));
                reminderDO.setTime(cursor.getString(3));
                reminderDO.setRepeat(cursor.getString(4));
                reminderDO.setRepeatNo(cursor.getString(5));
                reminderDO.setRepeatType(cursor.getString(6));
                reminderDO.setActive(cursor.getString(7));

                // Adding Reminders to list
                reminderDOList.add(reminderDO);
            } while (cursor.moveToNext());
        }
        return reminderDOList;
    }

    /**
     * This method is for Getting Reminders Count
     * @return reminder count
     */

    public int getRemindersCount() {
        String countQuery = "SELECT * FROM " + TABLE_REMINDERS;
        SQLiteDatabase db = this.getReadableDatabase();
        Cursor cursor = db.rawQuery(countQuery, null);
        cursor.close();

        return cursor.getCount();
    }

    /**
     * This method is for updating single ReminderDO
     * @param reminderDO
     * @return
     */

    public int updateReminder(ReminderDO reminderDO) {
        SQLiteDatabase db = this.getWritableDatabase();
        ContentValues values = new ContentValues();
        values.put(KEY_TITLE, reminderDO.getTitle());
        values.put(KEY_DATE, reminderDO.getDate());
        values.put(KEY_TIME, reminderDO.getTime());
        values.put(KEY_REPEAT, reminderDO.getRepeat());
        values.put(KEY_REPEAT_NO, reminderDO.getRepeatNo());
        values.put(KEY_REPEAT_TYPE, reminderDO.getRepeatType());
        values.put(KEY_ACTIVE, reminderDO.getActive());

        // Updating row
        return db.update(TABLE_REMINDERS, values, KEY_ID + "=?",
                new String[]{String.valueOf(reminderDO.getID())});
    }

    /**
     * This method is for deleting single ReminderDo
     * @param reminderDO
     */

    public void deleteReminder(ReminderDO reminderDO) {
        SQLiteDatabase db = this.getWritableDatabase();
        db.delete(TABLE_REMINDERS, KEY_ID + "=?",
                new String[]{String.valueOf(reminderDO.getID())});
        db.close();
    }
}
