from django.shortcuts import render, redirect
from django.views import View
from django.contrib.auth import authenticate, login, logout
from django.contrib import messages
from django.urls import reverse_lazy
from django.contrib.auth.decorators import login_required
from django.contrib.auth.mixins import LoginRequiredMixin, UserPassesTestMixin
from django.db import transaction
from django.http import JsonResponse
import json
from datetime import datetime
from .models import *
from django.contrib.auth.models import User
from .forms import *
from .allocator import SPA
from.LatexGen import mappedResult
from django_drf_filepond.api import store_upload, delete_stored_upload
from django_drf_filepond.models import TemporaryUpload, StoredUpload


# View to handle login page
class LoginView(View):
	template_name = 'login.html'
	
	def get(self,request,*args,**kwargs):
		# Render login page
		return render(request,self.template_name)

	def post(self,request,*args,**kwargs):

		# Get details from the post request
		username = request.POST.get('username')
		password = request.POST.get('password')

		user  = authenticate(request,username=username,password=password)

		if user is not None:
			# user exists
			login(request,user)
			return redirect('gap:dashboard')
		else:
			messages.error(request,'Username or password is incorrect')
			return render(request,self.template_name)


# Function to logout a user
@login_required(login_url = 'gap:login')
def LogoutView(request):
	logout(request)
	return redirect('gap:login')


# Dashboard for all users
class dashboardView(LoginRequiredMixin,UserPassesTestMixin,View):
	login_url = reverse_lazy('gap:login')
	redirect_field_name = 'next'

	def test_func(self):
		# used to set template name based on type of user
		self.user_type = get_user_type(self.request.user)
		if self.user_type == 'student':
			self.template_name = 'student_dashboard.html'
		else:
			self.template_name = 'faculty_dashboard.html'
		return True

	def get(self,request,*args,**kwargs):
		# Render dashboard based on type of user
		context= get_context(self.user_type,request)
		return render(request,self.template_name,context)

	def post(self,request,*args,**kwargs):
		if self.user_type == 'student':
			# Handling student post requests
			user_application = StudentDetailModel.objects.filter(student=request.user)
			if user_application.exists():
				# Update existing data
				user_application = user_application.first()
				form = StudentDetailModelForm(request.POST,instance=user_application)
				education_detail = StudentEducationFormset(request.POST,instance=user_application)
				project_detail = StudentProjectFormset(request.POST,instance=user_application)
			else:
				# Create new model row
				form = StudentDetailModelForm(request.POST)
				education_detail = StudentEducationFormset(request.POST)
				project_detail = StudentProjectFormset(request.POST)
			if form.is_valid():
				# Form validated
				self.object = form.save(commit=False)
				self.object.student = request.user
				# Save the main form , so it can be used for dependant models
				self.object.save()

				# Save education details of student
				with transaction.atomic():
					# Refer to the main model
					education_detail.instance = self.object
					if education_detail.is_valid():
						ids = get_ids(request.POST,'education')
						self.object.education.exclude(id__in=ids).delete()
						education_detail.save()
					else:
						messages.error(request,education_detail.errors,extra_tags='Education Details')

				# Save project details of student
				with transaction.atomic():
					# Refer to the main model
					project_detail.instance = self.object
					if project_detail.is_valid():
						ids = get_ids(request.POST,'projects')
						self.object.projects.exclude(id__in=ids).delete()
						project_detail.save()
					else:
						messages.error(request,project_detail.errors,extra_tags='Project Details')
				
				# Handle file upload 
				upload_id = request.POST.get('filepond')
				user_application = StudentDetailModel.objects.get(student=request.user)
				resume_record = StudentResumeModel.objects.filter(student=user_application)
				if resume_record and resume_record.count() != 0:
					# Handle already uploaded previous resume
					resume_record = resume_record.first()
					curr_upload_id = resume_record.file.upload_id 
					if upload_id == "":
						# Delete the resume if user has requested to delete it
						resume_record.delete()
						delete_stored_upload(curr_upload_id,delete_file=True)
					elif curr_upload_id != upload_id:
						# Update previous resume if new one is different from old resume
						temp_upload = TemporaryUpload.objects.get(upload_id=upload_id)
						file_reference = store_upload(upload_id,destination_file_path=str(request.user.username) + "/"+ temp_upload.upload_name )
						resume_record.file = file_reference
						resume_record.save()
						delete_stored_upload(curr_upload_id,delete_file=True)
				else:
					if upload_id != "":
						# Create new resume model row
						temp_upload = TemporaryUpload.objects.get(upload_id=upload_id)
						file_reference = store_upload(upload_id,destination_file_path=str(request.user.username) + "/"+ temp_upload.upload_name )
						resume_record = StudentResumeModel(student=user_application,file=file_reference)
						resume_record.save()
			else:
				messages.error(request,form.errors,extra_tags='main_form')
			if form.is_valid() and education_detail.is_valid() and project_detail.is_valid():
				return redirect('gap:dashboard')
			context = get_context(self.user_type,request)
			return render(request,self.template_name,context)
		else:

			# Handle faculty and facad users

			if request.POST.get('positionId'):
				# Update the already present open positions
				position = PositionsModel.objects.get(id=request.POST.get('positionId'))
				form = PositionsModelForm(request.POST,instance=position)
				requisite_lists = RequisitesFormset(request.POST,instance=position)
			else:
				# Create a new open position
				form = PositionsModelForm(request.POST)
				requisite_lists = RequisitesFormset(request.POST)
			errors = form.errors
			if form.is_valid():
				self.object = form.save(commit=False)
				self.object.faculty = request.user
				self.object.initial_total_positions = self.object.total_positions
				# Save the main form , so it can be used for dependant models
				self.object.save()

				# Save requisite details of project
				with transaction.atomic():
					# Refer to main model
					requisite_lists.instance = self.object
					if requisite_lists.is_valid():
						ids = get_ids(request.POST,'requisites')
						self.object.requisites.exclude(id__in=ids).delete()
						requisite_lists.save()
					else:
						messages.error(request,requisite_lists.errors,extra_tags='Requisite details')
			else:
				messages.error(request,form.errors,extra_tags='Form Details')
			if form.is_valid() and requisite_lists.is_valid():
				return redirect('gap:dashboard')
			messages.error(request,errors)
			context = get_context(self.user_type,request)
			return render(request,self.template_name,context)
				






# Helper codes
def get_preferences(pref_type):
	'''
	Input:
	pref_type: Student, faculty

	Output:
	Ordered list of priorities pertaining to all the users of that type
	'''
	result = {}
	emptyresult = {}
	counts = {}
	if pref_type=='student':
		students = StudentPriorities.objects.values_list('student',flat=True).distinct()
		for student in students:
			emptyresult[student] = []
			result[student] = list(StudentPriorities.objects.filter(student=student).order_by('current_priority').values_list('project',flat=True))
		return result, emptyresult
	else:
		projects = ProjectPriorities.objects.values_list('project',flat=True).distinct()
		for project in projects:
			emptyresult[project] = []
			counts[project] = PositionsModel.objects.get(id=project).total_positions
			result[project]  = list(ProjectPriorities.objects.filter(project=project).order_by('current_priority').values_list('student',flat=True))
		return result, emptyresult , counts

def get_user_type(user):
	'''
	Input:
	User: user object

	Output:
	Type of user
	'''
	if is_faculty(user):
		if is_facad(user):
			return 'facad'
		return 'faculty'
	return 'student'


def is_faculty(user):
	# Function to check if user is a faculty
    return user.groups.filter(name='faculty').exists()

def is_facad(user):
	# Function to check if user is  a facad
    return user.groups.filter(name='facad').exists()


def get_context(user_type,request):
	context = {}
	if not RoundDetailsModel.objects.all().exists():
		# Create a new round if it does not exist
		RoundDetailsModel().save()

	# Get current round details
	current_round = RoundDetailsModel.objects.latest('roundNo')
	context['current_round'] = current_round
	if user_type == 'student':
		print(request.POST)
		# Get student context details

		# User application with all student details
		user_application = StudentDetailModel.objects.filter(student=request.user)
		print(user_application)
		if user_application.exists():
			# All application details and prefilled forms
			context['applicationDetails'] = user_application.first()
			context['projectDetails'] = user_application.first().projects.all()
			context['educationDetails'] = user_application.first().education.all()
			context['ResumeDetails'] = user_application.first().resume.first()
			context['studentDetailForm'] = StudentDetailModelForm(None,instance=user_application.first())
			context['studentProjects'] = StudentProjectFormset(None,instance=user_application.first())
			context['studentEducation'] = StudentEducationFormset(None,instance=user_application.first())
			selected_vals = []
			for project in user_application.first().projects.all():
				selected_vals.append(project.languages)
			context['selected_project_languagues'] = selected_vals
		else:
			# Empty forms as user has not created his profile
			context['studentDetailForm'] = StudentDetailModelForm()
			context['studentProjects'] = StudentProjectFormset()
			context['studentEducation'] = StudentEducationFormset()



		# Prefilled forms
		context['user_type'] = user_type
		context['requisitesForm'] = RequisitesFormset()
		context['newPosition'] = PositionsModelForm()
		context['availablePositions'] = PositionsModel.objects.filter(faculty=request.user)
		updateforms = []
		updaterequisiteforms = []

		# List of all projects proposed by faculty
		for position in context['availablePositions']:
			updateforms.append(PositionsModelForm(None,instance=position))
			updaterequisiteforms.append(RequisitesFormset(None,instance=position))
		context['updateforms'] = updateforms
		context['updaterequisiteforms'] = updaterequisiteforms


	return context

def get_ids(data,prefix):
	# Get a list of all ids from post request based on specific prefix
	count = data[prefix + '-TOTAL_FORMS']
	ids = []
	for index in range(int(count)):
		try:
			curr_id = data[prefix + '-{}-id'.format(index)]
			if curr_id: 
				ids.append(int(curr_id))
		except:
			break
	return ids

