from django import template
from ..choices import language_color_mappings, color_choices
import random
from datetime import datetime, timezone

register = template.Library()


# Function to get particular element from a list
@register.filter
def index(indexable, i):
	'''
	Input:
	indexable: Array type date
	i: index

	Output:
	element at index i in indexable
	'''
	return indexable[i]

# Function to get color for a language
@register.simple_tag
def get_color(language):
	'''
	Input:
	language: Language

	Output:
	color for language
	'''
	return language_color_mappings[language]



# Function to get normalized grades
@register.simple_tag
def get_normalized_grades(grade):
	'''
	Input:
	grade: Unnormalized grade, can be for 100 or 10
	
	Output:
	grade normalized to 5
	'''
	if grade < 11:
		return grade / 2.0
	else:
		return grade / 20.0


# Function to get color for priority tables to guide
@register.simple_tag
def get_table_color(index):
	'''
	Input:
	index: Index of the table'

	Output:
	Random color choice
	'''
	index = index % len(color_choices)
	return color_choices[index]

# Function to get random color
@register.simple_tag
def get_random_color(index):
	'''
	Input:
	index: Index 

	Output:
	Random color choice
	'''
	return random.choice(color_choices)

# Function to get cout of elements in nested list
@register.filter
def get_count_multilist(indexable):
	'''
	Input:
	indexable: Nested list

	Output:
	count of elements in indexable
	'''
	return flatten_count(indexable)


# Function to calculate left over days for round ending
@register.simple_tag
def get_remaining_days(date):
	'''
	Input: 
	date: roundDetail object

	Output:
	string showing remaining days
	'''
	string = ''
	now = datetime.now(timezone.utc).date()
	diff = date - now

	year = diff.days//365.25
	month = (diff.days%365) // 30
	day = (diff.days%30)
	hour = diff.seconds//3600
	minute = (diff.seconds%3600) // 60
	second = diff.seconds%60
	if year != 0 :
		if year == 1:
			string = '{} year to go'.format(year)
		else:
			string = '{} years to go'.format(year)  
	elif month != 0 :
		if month == 1:
			string = '{} month to go'.format(month)  
		else:
			string = '{} months to go'.format(month) 
	elif day != 0 :
		if day == 1:
			string = '{} day to go'.format(day)
		else:
			string = '{} days to go'.format(day)
	elif hour != 0 :
		if hour == 1:
			string = '{} hour to go'.format(hour)
		else:
			string = '{} hours to go'.format(hour)
	elif minute != 0 :
		if minute == 1:
			string = '{} minute to go'.format(minute)
		else:
			string = '{} minutes to go'.format(minute)
	else:
		if second == 0:
			string = ' About to end'
		elif second == 1:
			string = '{} second to go'.format(second)
		else:
			string = '{} seconds to go'.format(second)
	return string


# Function to get percentage of time left over
@register.simple_tag
def get_progress_percent(current_round):
	'''
	Input: 
	date: roundDetail object

	Output:
	percentage of remaining time
	'''
	now = datetime.now(timezone.utc).date()
	start_date = current_round.startDate
	end_date = current_round.endDate
	return (end_date - now).days*100.0 / (end_date-start_date).days

# Function to get color of timebar
@register.simple_tag
def get_progress_color(date):
	'''
	Input:
	Date : end date

	Output:
	returns color based  on remaining time
	'''
	now = datetime.now(timezone.utc).date()
	diff = date - now
	if diff.days >= 6:
		return "green"
	if diff.days >4:
		return "yellow"
	if diff.days >2:
		return "orange"
	return "red"

# Helper codes

# COde to get count of nested list
def flatten_count(listOrItem, result = None):
	if type(listOrItem) != type([]):    # Handle non-list by appending.
		return 1
	else:
		count = 0
		for item in listOrItem:         # Recursively handle each item in a list.
			count+=flatten_count(item, result)
		return count