from django.db import models
from django_drf_filepond.models import StoredUpload
from django.core.validators import MinValueValidator, MaxValueValidator
from django.contrib.auth.models import User
from django.conf import settings

from .choices import *
from multiselectfield import MultiSelectField
from datetime import datetime,timedelta
import os


# Model to store the project details
class PositionsModel(models.Model):
	faculty = models.ForeignKey(User, on_delete=models.DO_NOTHING, null=False, 
								blank=False,default=None, related_name='position_user')
	title = models.CharField(verbose_name="Project Title",max_length=100,blank=False,null=False)
	description = models.TextField(blank=True,null=True,verbose_name="Description")
	contact_details = models.CharField(verbose_name="Contact Details",max_length=200,null=True,blank=True)
	total_positions = models.IntegerField(verbose_name="Total Positions",null=False,blank=False)	
	initial_total_positions = models.IntegerField(verbose_name="Initial Total Positions",null=True,blank=True)
	def get_requisites(self):
		# Gives the list of all requisites for particular project
		return self.requisites.all()

	def get_alloted_students(self):
		# Gives list of all students alloted to this project
		return self.student_mappings.all()

# Model to store requisites for every project
class RequisitesModel(models.Model):
	position = models.ForeignKey(PositionsModel,related_name='requisites',on_delete=models.CASCADE)

	Subject = models.CharField(verbose_name="Subject Name",max_length=6,choices=subject_choices,default='CS699')
	cgpa = models.FloatField(verbose_name="CGPA",null=False,blank=False,
								validators=[MinValueValidator(0.0),MaxValueValidator(10.0)])

# Model to store details of student profile
class StudentDetailModel(models.Model):
	student = models.ForeignKey(User, on_delete=models.DO_NOTHING, null=False, 
								blank=False,default=None, related_name='student_user')
	qualities = models.CharField(verbose_name="Description",max_length=200,blank=True,null=True)
	alloted_project = models.ForeignKey(PositionsModel,on_delete=models.DO_NOTHING,blank=True,default=None,null=True,related_name="alloted_student")

	def get_education_details(self):
		# Gives the list of all education details for student
		return self.education.all()
	
	def get_project_details(self):
		# Gives the list of all project details for student
		return self.projects.all()

	def get_resume_details(self):
		# Gives the resume associated with the student
		return self.resume.first()

	def get_cgpa(self):
		# Gives the BE/Btech cgpa of the student the student
		if self.education.exists():
			return self.education.get(exam='BE/BTech').cgpa
		else:
			return 0

	def get_alloted_project(self):
		if self.student_allotment:
			return self.student_allotment.project.title
		else:
			return None

# Model to store details of student projects
class StudentProjectsModel(models.Model):
	student = models.ForeignKey(StudentDetailModel,related_name='projects',on_delete=models.CASCADE)
	title = models.CharField(verbose_name="Project Title",max_length=100,blank=False,null=False)
	languages = MultiSelectField(verbose_name="Programming Languages",choices=language_choices)
	description = models.TextField(verbose_name="Description",blank=True,null=True)
	link = models.URLField(verbose_name="Github Link",max_length=200,blank=True,null=True)

# Model to store details of student education
class StudentEducationModel(models.Model):
	student = models.ForeignKey(StudentDetailModel,related_name='education',on_delete=models.CASCADE)
	exam = models.CharField(verbose_name="Examination",max_length=10,choices=exam_choices,default="BE/BTech")
	board = models.CharField(verbose_name="Examination Board",max_length=100,blank=False,null=False)
	cgpa = models.FloatField(verbose_name="CGPA/Percentage",null=False,blank=False,
								validators=[MinValueValidator(0.0),MaxValueValidator(100.0)])
	year = models.DateField(verbose_name="Year of completion",null=False,blank=False)


# Model to store details of student resume
class StudentResumeModel(models.Model):
	student = models.ForeignKey(StudentDetailModel,related_name="resume",on_delete=models.CASCADE)
	file = models.ForeignKey(StoredUpload,on_delete=models.DO_NOTHING)

# Model to store priorities of student for ongoing round
class StudentPriorities(models.Model):
	student = models.ForeignKey(StudentDetailModel,related_name='priorities',on_delete=models.CASCADE,db_column="student_id")
	project = models.ForeignKey(PositionsModel,related_name='student_priorities',on_delete=models.CASCADE,db_column="project_id")
	current_priority = models.IntegerField(verbose_name="Priority Value",null=False,blank=False,default=-1)
	date = models.DateField(verbose_name="Application Date",null=False,blank=False,default=datetime.now)

# Model to store priorities of student for previous rounds
class StudentPrioritiesArchive(models.Model):
	roundNo = models.IntegerField(verbose_name="Round Number",null=False,blank=False,default=1)
	student = models.ForeignKey(StudentDetailModel,related_name='archived_priorities',on_delete=models.CASCADE,db_column="student_id")
	project = models.ForeignKey(PositionsModel,related_name='archived_student_priorities',on_delete=models.CASCADE,db_column="project_id")
	current_priority = models.IntegerField(verbose_name="Priority Value",null=False,blank=False,default=-1)
	date = models.DateField(verbose_name="Application Date",null=False,blank=False,default=datetime.now)

# Model to store priorities of project for ongoing round
class ProjectPriorities(models.Model):
	project = models.ForeignKey(PositionsModel,related_name='priorities',on_delete=models.CASCADE,db_column="project_id")
	student = models.ForeignKey(StudentDetailModel,related_name='project_priorities',on_delete=models.CASCADE,db_column="student_id")
	current_priority = models.IntegerField(verbose_name="Priority Value",null=False,blank=False,default=-1)
	date = models.DateField(verbose_name="Application Date",null=False,blank=False,default=datetime.now)

# Model to store priorities of project for previous rounds
class ProjectPrioritiesArchive(models.Model):
	roundNo = models.IntegerField(verbose_name="Round Number",null=False,blank=False,default=1)
	project = models.ForeignKey(PositionsModel,related_name='archived_priorities',on_delete=models.CASCADE,db_column="project_id")
	student = models.ForeignKey(StudentDetailModel,related_name='archived_project_priorities',on_delete=models.CASCADE,db_column="student_id")
	current_priority = models.IntegerField(verbose_name="Priority Value",null=False,blank=False,default=-1)
	date = models.DateField(verbose_name="Application Date",null=False,blank=False,default=datetime.now)

# Model to store the details of every round
class RoundDetailsModel(models.Model):
	roundNo = models.IntegerField(verbose_name="Round Number",null=False,blank=False,default=1,primary_key = True)
	startDate = models.DateField(verbose_name="Round Start Date",null=False,blank=False,default=datetime.now)
	endDate = models.DateField(verbose_name="Round End Date",null=False,blank=False,default=datetime.now()+timedelta(days=10))
	allocated = models.IntegerField(verbose_name="Count of Alloted students",null=True,blank=True)
	unallocated = models.IntegerField(verbose_name="Count of Unalloted students",null=True,blank=True)

# Model to store generated mappings for every round
class GeneratedMappingsModel(models.Model):
	roundNo = models.IntegerField(verbose_name="Round Number",null=False,blank=False,default=1)
	project = models.ForeignKey(PositionsModel,related_name="student_mappings",on_delete=models.DO_NOTHING,db_column="project_id")
	faculty = models.ForeignKey(User,on_delete=models.DO_NOTHING,db_column="faculty_id")
	student = models.ForeignKey(StudentDetailModel,related_name="student_allotment",on_delete=models.DO_NOTHING,db_column="student_id")