/*
 * mm-naive.c - The fastest, least memory-efficient malloc package.
 *
 * In this naive approach, a block is allocated by simply incrementing
 * the brk pointer.  A block is pure payload. There are no headers or
 * footers.  Blocks are never coalesced or reused. Realloc is
 * implemented directly using mm_malloc and mm_free.
 *
 * NOTE TO STUDENTS: Replace this header comment with your own header
 * comment that gives a high level description of your solution.
 */
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>
#include <string.h>
#include<stddef.h>
#include<stdint.h>

#include "mm.h"
#include "memlib.h"
#include "config.h"

/*********************************************************
 * NOTE TO STUDENTS: Before you do anything else, please
 * provide your team information in the following struct.
 ********************************************************/
team_t team = {
    /* Team name */
    "team name",
    /* First member's full name */
    "member 1",
    /* First member's email address */
    "member_1@cse.iitb.ac.in",
    /* Second member's full name (leave blank if none) */
    "member 2",
    /* Second member's email address (leave blank if none) */
    "member_2@cse.iitb.ac.in"
};


typedef struct free_node_
{
struct free_node_ *prev_node;

struct free_node_ *next_node;

}free_node_t;

typedef enum
{
    MM_FALSE,
    MM_TRUE
} vm_bool_t;


typedef struct block_meta_data_
{
    vm_bool_t is_free;
    uint32_t block_size;
    struct block_meta_data_ *prev_block;
    struct block_meta_data_ *next_block;
    free_node_t free_node;
} block_meta_data_t;

typedef struct block_end_meta_data_
{
uint32_t block_size;

} block_end_meta_data_t;

/* single word (4) or double word (8) alignment */
#define ALIGNMENT 8

/* rounds up to the nearest multiple of ALIGNMENT */
#define ALIGN(size) (((size) + (ALIGNMENT-1)) & ~0x7)


#define SIZE_T_SIZE (ALIGN(sizeof(size_t)))

#define NEXT_META_BLOCK(block_meta_data_ptr)  \
       (block_meta_data_ptr->next_block)

#define NEXT_META_BLOCK_BY_SIZE(block_meta_data_ptr) \
        (block_meta_data_t *)((char *)(block_meta_data_ptr)+ALIGN(sizeof(block_meta_data_t))  \
        + block_meta_data_ptr->block_size \
        +ALIGN(sizeof(block_end_meta_data_t)))

#define PREV_META_BLOCK(block_meta_data_ptr) \
        (block_meta_data_ptr->prev_block)

#define mm_bind_blocks_for_allocation(allocated_meta_block,free_meta_block) \
    free_meta_block->prev_block=allocated_meta_block; \
    free_meta_block->next_block=allocated_meta_block->next_block; \
    allocated_meta_block->next_block=free_meta_block;   \
    if(free_meta_block->next_block) \
    free_meta_block->next_block->prev_block=free_meta_block

#define meta_data_block_addr(free_node_addr) \
    (block_meta_data_t *) ((char *)(free_node_addr)-offsetof(block_meta_data_t,free_node))

#define end_meta_data_addr(meta_data_addr) \
    (block_end_meta_data_t *)((char *)(meta_data_addr)+ALIGN(sizeof(block_meta_data_t)) \
    +meta_data_addr->block_size)

#define start_meta_data_addr(end_meta_data_addr) \
    (block_meta_data_t *)((char *)(end_meta_data_addr) \
    -end_meta_data_addr->block_size \
    -ALIGN(sizeof(block_meta_data_t)))




#define TOTAL_COMBINE_SIZE_OF_BLOCKS(base)  \
    base->block_size \
    +ALIGN(sizeof(block_end_meta_data_t)) \
    +ALIGN(sizeof(block_meta_data_t))

#define TOTAL_COMBINED_SIZE_OF_BLOCKS(base) \
    base->next_block->block_size \
    +base->prev_block->block_size \
    +2*ALIGN(sizeof(block_end_meta_data_t)) \
    +2*ALIGN(sizeof(block_meta_data_t))


#define META_DATA_OVERHEAD  \
    (uint32_t)(ALIGN(sizeof(block_end_meta_data_t))+ALIGN(sizeof(block_meta_data_t)))

/*
 * mm_init - initialize the malloc package.
 */

void *init_mem_sbrk_break = NULL;

free_node_t **head=NULL;

free_node_t init_free_node(void)
{
    free_node_t freenode;
    freenode.next_node=NULL;
    freenode.prev_node=NULL;
    return freenode;
}


void insert_node_in_freelist(free_node_t **head,block_meta_data_t *new_metadata_block)
{

    free_node_t *prev_node=NULL;
    free_node_t *next_node=NULL;

    free_node_t *temp=*head;

    int count=0;

    int flag=0;


    while(temp!=NULL)
    {

        block_meta_data_t *meta_data=meta_data_block_addr(temp);

       if(new_metadata_block->block_size>=meta_data->block_size)
       {

            if(temp->next_node!=NULL)
            {
            temp=temp->next_node;
            }

            else
            {
                temp->next_node=&(new_metadata_block->free_node);
                new_metadata_block->free_node.prev_node=temp;
                flag=1;
                break;
            }
       }

       else
       {
        prev_node=temp->prev_node;
        next_node=temp;
        break;
       }
    }

    if(flag==0)
    {
    if(temp==*head)
    {
        *head=&(new_metadata_block->free_node);
    }

    new_metadata_block->free_node.next_node=next_node;
    if(next_node)
        next_node->prev_node = &(new_metadata_block->free_node);
    new_metadata_block->free_node.prev_node=prev_node;
    if(prev_node)
        prev_node->next_node = &(new_metadata_block->free_node);
    }
}

block_meta_data_t *get_node_from_freelist(free_node_t **head,uint32_t reqsize)
{
   free_node_t *temp=*head;

   block_meta_data_t *meta_data=NULL;
   while(temp!=NULL)
   {

    meta_data=meta_data_block_addr(temp);

    if(meta_data->block_size<reqsize)
    {
        temp=temp->next_node;
    }

    else
    {
        return meta_data;
    }

   }
   return NULL;
}

void merge_free_blocks(block_meta_data_t *first,block_meta_data_t *second)
{

    assert(first->is_free==MM_TRUE&&second->is_free==MM_TRUE);

    first->block_size+=ALIGN(sizeof(block_meta_data_t)) + second->block_size+ALIGN(sizeof(block_end_meta_data_t));

    first->next_block=second->next_block;

    if(second->next_block!=NULL)
    {
    second->next_block->prev_block=first;
    }


}


void remove_block_from_free_list(free_node_t **head,free_node_t *del_node)
{

    if(del_node==*head)
    {

        if(del_node->next_node!=NULL)
        {
        del_node->next_node->prev_node=NULL;
        }
        *head=del_node->next_node;
    }

    else
    {
    //printf("Came here\n");
    free_node_t *temp=*head;

    while(temp->next_node!=del_node)
    {
        temp=temp->next_node;
    }

    temp->next_node=del_node->next_node;

    if(del_node->next_node!=NULL)
    {
        del_node->next_node->prev_node=temp;
    }
    }
    del_node->prev_node=NULL;
    del_node->next_node=NULL;
    // while (head1)
    // {
    //     block_meta_data_t *trace = (block_meta_data_t *)meta_data_block_addr(head1);
    //     printf("%d [FREE BLOCK SIZE]\n",trace->block_size);
    //     head1 = head1->next_node;
    // }
}

int get_internal_fragmented_size(block_meta_data_t *block1,block_meta_data_t *block2)
{

    block_meta_data_t *next_meta_block_by_size=NEXT_META_BLOCK_BY_SIZE(block1);

    return (int)((char *)block2-(char *)next_meta_block_by_size);

}

void *allocate_requested_size(size_t size)
{
    size_t sys_page_size=mem_pagesize();
    unsigned int no_of_pages_req=((size+2*ALIGN(sizeof(block_end_meta_data_t))+2*ALIGN(sizeof(block_meta_data_t))+sys_page_size-1)/sys_page_size);
    size_t extra_size_req=no_of_pages_req*sys_page_size;
    void *store_brk = mem_heap_hi();
    void *brk= mem_sbrk(extra_size_req);
    if(store_brk==brk && *(int *)brk == -1)
        return NULL;
    return brk;
}

void create_new_block_and_insert_into_freelist(block_meta_data_t *meta_data_block,uint32_t final_remain)
{
    block_end_meta_data_t *end_meta_data_block = (block_end_meta_data_t *)end_meta_data_addr(meta_data_block);
    end_meta_data_block->block_size = meta_data_block->block_size;
    if(meta_data_block->is_free == MM_TRUE)
        meta_data_block->is_free = MM_FALSE;
    if(final_remain < META_DATA_OVERHEAD)
    {
        if(meta_data_block->next_block)
            meta_data_block->next_block->prev_block = meta_data_block;
        return;
    }
    block_meta_data_t *new_block = (block_meta_data_t *)NEXT_META_BLOCK_BY_SIZE(meta_data_block);
    new_block->block_size = final_remain-META_DATA_OVERHEAD;
    new_block->free_node = init_free_node();
    new_block->is_free = MM_TRUE;
    mm_bind_blocks_for_allocation(meta_data_block, new_block);
    assert(meta_data_block->next_block->block_size == new_block->block_size);
    block_end_meta_data_t *new_end_meta_data_block = (block_end_meta_data_t *)end_meta_data_addr(new_block);
    new_end_meta_data_block->block_size = new_block->block_size;
    if(new_block->next_block && new_block->next_block->is_free == MM_TRUE)
        merge_free_blocks(new_block, new_block->next_block);
    if(new_block->prev_block && new_block->prev_block->is_free == MM_TRUE)
        merge_free_blocks(new_block->prev_block,new_block);
    insert_node_in_freelist(head,new_block);
}

int mm_init(void)
{

    if(init_mem_sbrk_break==NULL)
    {
    mem_init();
    }
    mem_reset_brk();
    init_mem_sbrk_break=mem_sbrk(mem_pagesize());
    if(*(int *)init_mem_sbrk_break==-1)
    {
        return -1;
    }
    head=(free_node_t **)init_mem_sbrk_break;
    block_meta_data_t *metadata=(block_meta_data_t *)((char *)head+ALIGN(sizeof(head)));
    metadata->is_free=MM_TRUE;
    metadata->block_size=mem_pagesize()-ALIGN(sizeof(block_meta_data_t))-ALIGN(sizeof(block_end_meta_data_t))-ALIGN(sizeof(head));
    metadata->free_node=init_free_node();
    metadata->next_block=NULL;
    metadata->prev_block=NULL;
    *head=&(metadata->free_node);
    block_end_meta_data_t *end_meta_data=end_meta_data_addr(metadata);
    end_meta_data->block_size=metadata->block_size;
	//This function is called every time before each test run of the trace.
	//It should reset the entire state of your malloc or the consecutive trace runs will give wrong answer.


	/*
	 * This function should initialize and reset any data structures used to represent the starting state(empty heap)
	 *
	 * This function will be called multiple time in the driver code "mdriver.c"
	 */

    return 0;		//Returns 0 on successfull initialization.
}

//---------------------------------------------------------------------------------------------------------------
/*
 * mm_malloc - Allocate a block by incrementing the brk pointer.
 *     Always allocate a block whose size is a multiple of the alignment.
 */
void *mm_malloc(size_t size)
{
	/*
	 * This function should keep track of the allocated memory blocks.
	 * The block allocation should minimize the number of holes (chucks of unusable memory) in the heap memory.
	 * The previously freed memory blocks should be reused.
	 * If no appropriate free block is available then the increase the heap  size using 'mem_sbrk(size)'.
	 * Try to keep the heap size as small as possible.
	 */

    printf("%u [IN MALLOC]\n", size);
	if(size <= 0){		// Invalid request size
		return NULL;
	}
	size = ALIGN(size);		//size alligned to 8 bytes

     block_meta_data_t *free_metadata_block=get_node_from_freelist(head,size);

    //If No free block available then we will request new pages by calling mem_sbrk()
     if(free_metadata_block==NULL)
     {

       void *brk = allocate_requested_size(size);
    
        
       if(brk==NULL)
       {
        printf("RETURNINH IN HERE\n");
        return brk;
       }

       block_end_meta_data_t *prev_end_meta_data=NULL;

       prev_end_meta_data=(block_end_meta_data_t *)((char *)brk-ALIGN(sizeof(block_end_meta_data_t)));

        block_meta_data_t *prev_start_meta_data=NULL;

        prev_start_meta_data=start_meta_data_addr(prev_end_meta_data);

        block_meta_data_t *next_meta_data_block=(block_meta_data_t *)brk;

        next_meta_data_block->is_free=MM_FALSE;

        next_meta_data_block->block_size=size;

        next_meta_data_block->free_node=init_free_node();

        next_meta_data_block->prev_block=prev_start_meta_data;

        prev_start_meta_data->next_block=next_meta_data_block;

        next_meta_data_block->next_block=NULL;

        block_end_meta_data_t *next_end_meta_data_block=end_meta_data_addr(next_meta_data_block);

        next_end_meta_data_block->block_size=next_meta_data_block->block_size;

        block_meta_data_t *new_free_meta_block=NEXT_META_BLOCK_BY_SIZE(next_meta_data_block);

        new_free_meta_block->is_free=MM_TRUE;

        // new_free_meta_block->block_size=extra_size_req-size-2*sizeof(block_meta_data_t)-(2*sizeof(block_end_meta_data_t));

        new_free_meta_block->block_size= (uint32_t)((char *)(mem_heap_hi()+1)-(char *)brk-size-(2*(uint32_t)ALIGN(sizeof(block_meta_data_t)))-(2*(uint32_t)ALIGN(sizeof(block_end_meta_data_t))));

        new_free_meta_block->free_node=init_free_node();

        new_free_meta_block->prev_block=NULL;

        block_end_meta_data_t *new_end_free_meta_block=end_meta_data_addr(new_free_meta_block);

        new_end_free_meta_block->block_size=new_free_meta_block->block_size;

        insert_node_in_freelist(head,new_free_meta_block);

        mm_bind_blocks_for_allocation(next_meta_data_block,new_free_meta_block);

        return (void *)((char *)next_meta_data_block+ALIGN(sizeof(block_meta_data_t)));
     }

     assert(free_metadata_block->is_free==MM_TRUE);

     remove_block_from_free_list(head,&free_metadata_block->free_node);

     uint32_t remaining_size=free_metadata_block->block_size-size;

     free_metadata_block->is_free=MM_FALSE;

     free_metadata_block->block_size=size;

     block_end_meta_data_t *end_meta_data=end_meta_data_addr(free_metadata_block);

     end_meta_data->block_size=size;


    //Here we will created new block by splitting previous block
     if(ALIGN(sizeof(block_meta_data_t))+ALIGN(sizeof(block_end_meta_data_t))<remaining_size){

        block_meta_data_t *next_meta_block=NULL;

        block_end_meta_data_t *next_end_meta_block=NULL;

        next_meta_block=NEXT_META_BLOCK_BY_SIZE(free_metadata_block);

        next_meta_block->is_free=MM_TRUE;

        next_meta_block->block_size=remaining_size-ALIGN(sizeof(block_meta_data_t))-ALIGN(sizeof(block_end_meta_data_t));

        next_meta_block->free_node=init_free_node();

        next_end_meta_block=end_meta_data_addr(next_meta_block);

        next_end_meta_block->block_size=next_meta_block->block_size;

        insert_node_in_freelist(head,next_meta_block);

        mm_bind_blocks_for_allocation(free_metadata_block,next_meta_block);

     }



     return (void *)((char *)free_metadata_block+ALIGN(sizeof(block_meta_data_t)));

    //mem_sbrk() is wrapper function for the sbrk() system call.
    //Please use mem_sbrk() instead of sbrk() otherwise the evaluation results
    //may give wrong results
}


void mm_free(void *ptr)
{


	/*
	 * Searches the previously allocated node for memory block with base address ptr.
	 *
	 * It should also perform coalesceing on both ends i.e. if the consecutive memory blocks are
	 * free(not allocated) then they should be combined into a single block.
	 *
	 * It should also keep track of all the free memory blocks.
	 * If the freed block is at the end of the heap then you can also decrease the heap size
	 * using 'mem_sbrk(-size)'.
	 */
    printf("[IN FREE]\n");
     block_meta_data_t *meta_data_block=(block_meta_data_t *)((char *)ptr-ALIGN(sizeof(block_meta_data_t)));

     assert(meta_data_block->is_free==MM_FALSE);

     meta_data_block->is_free=MM_TRUE;

     block_meta_data_t *prev_meta_data_block=meta_data_block->prev_block;

     block_meta_data_t *next_meta_data_block=meta_data_block->next_block;

     block_meta_data_t *merged_meta_data_block=NULL;

     block_end_meta_data_t *merged_end_meta_data_block=NULL;


     if(next_meta_data_block!=NULL)
     {
           meta_data_block->block_size+= get_internal_fragmented_size(meta_data_block,next_meta_data_block);
     }
     //If block being free is upper most block
    else
    {
        int fragmented_size=(int)((char *)mem_heap_hi()+1-((char *)end_meta_data_addr(meta_data_block)+ALIGN(sizeof(block_end_meta_data_t))));
        meta_data_block->block_size+=fragmented_size;
    }

    merged_meta_data_block=meta_data_block;

    if(next_meta_data_block!=NULL&&next_meta_data_block->is_free==MM_TRUE)
    {
        remove_block_from_free_list(head,&next_meta_data_block->free_node);

        merge_free_blocks(meta_data_block,next_meta_data_block);

        merged_meta_data_block=meta_data_block;

    }

    if(prev_meta_data_block!=NULL&&prev_meta_data_block->is_free==MM_TRUE)
    {
        remove_block_from_free_list(head,&prev_meta_data_block->free_node);

        prev_meta_data_block->block_size += get_internal_fragmented_size(prev_meta_data_block,meta_data_block);

        merge_free_blocks(prev_meta_data_block,meta_data_block);

        merged_meta_data_block=prev_meta_data_block;
    }

    merged_end_meta_data_block=(block_end_meta_data_t *)(end_meta_data_addr(merged_meta_data_block));

    merged_end_meta_data_block->block_size=merged_meta_data_block->block_size;

    insert_node_in_freelist(head,merged_meta_data_block);
}



/*
 * mm_realloc - Implemented simply in terms of mm_malloc and mm_free
 */
void *mm_realloc(void *ptr, size_t size)
{

    /*
	 * This function should also copy the content of the previous memory block into the new block.
	 * You can use 'memcpy()' for this purpose.
	 *
	 * The data structures corresponding to free memory blocks and allocated memory
	 * blocks should also be updated.
	*/

    printf("[IN REALLOC]\n");
	size = ((size+7)/8)*8; //8-byte alignement

    // printf("%lu [SIZE AFTER ALIGNMNET]",size);

	if(ptr == NULL){			//memory was not previously allocated
		return mm_malloc(size);
	}

	if(size == 0){				//new size is zero
		mm_free(ptr);
		return NULL;
	}

    block_meta_data_t *meta_data_block=(block_meta_data_t *)((char *)ptr-ALIGN(sizeof(block_meta_data_t)));

    if(size == meta_data_block->block_size)
        return meta_data_block;


    // if provided size is less than already allocated size `if block` will be executed otherwiae `else block`.
    if(size < meta_data_block->block_size)
    {
        uint32_t original_size = (uint32_t)meta_data_block->block_size;
        uint32_t inter_frag = (uint32_t)get_internal_fragmented_size(meta_data_block,meta_data_block->next_block);
        meta_data_block->block_size = size;
        block_end_meta_data_t *end_meta_data_block = (block_end_meta_data_t *)end_meta_data_addr(meta_data_block);
        end_meta_data_block->block_size = meta_data_block->block_size;
        if(meta_data_block->next_block && meta_data_block->next_block->is_free==MM_TRUE)
        {
            block_meta_data_t *new_block = NEXT_META_BLOCK_BY_SIZE(meta_data_block);
            remove_block_from_free_list(head, &meta_data_block->next_block->free_node);
            new_block->block_size = meta_data_block->next_block->block_size + inter_frag + (original_size - size);
            meta_data_block->next_block = meta_data_block->next_block->next_block;
            new_block->is_free = MM_TRUE;
            new_block->free_node= init_free_node();
            mm_bind_blocks_for_allocation(meta_data_block, new_block);

            block_end_meta_data_t *new_end_meta_data_block = (block_end_meta_data_t *)end_meta_data_addr(new_block);
            new_end_meta_data_block->block_size = new_block->block_size;
            insert_node_in_freelist(head,new_block);

            return (void *)((char *)meta_data_block+ALIGN(sizeof(block_meta_data_t)));

        }
        else
        {
            uint32_t inter_frag =0;
            int fragmented_size=(int)((char *)mem_heap_hi()+1-((char *)end_meta_data_addr(meta_data_block)+ALIGN(sizeof(block_end_meta_data_t))));
            if(!meta_data_block->next_block){
                if( (fragmented_size+original_size-size) < (ALIGN(sizeof(block_end_meta_data_t))+ALIGN(sizeof(end_meta_data_block))) )
                    return meta_data_block;
            }
            else{
                uint32_t inter_frag = get_internal_fragmented_size(meta_data_block,meta_data_block->next_block);
                if( (inter_frag+original_size-size) < (ALIGN(sizeof(block_end_meta_data_t))+ALIGN(sizeof(end_meta_data_block))) )
                    return meta_data_block;
                fragmented_size = 0;
            }

            meta_data_block->block_size  = size;
            block_end_meta_data_t *end_meta_data_block = (block_end_meta_data_t *)end_meta_data_addr(meta_data_block);
            end_meta_data_block->block_size = meta_data_block->block_size;

            if((original_size- size + inter_frag + fragmented_size) >= META_DATA_OVERHEAD)
            {
                block_meta_data_t *new_block = (block_meta_data_t *)NEXT_META_BLOCK_BY_SIZE(meta_data_block);
                new_block->free_node = init_free_node();
                new_block->is_free = MM_TRUE;
                new_block->block_size = original_size- size + inter_frag + fragmented_size - META_DATA_OVERHEAD;
                mm_bind_blocks_for_allocation(meta_data_block, new_block);
                block_end_meta_data_t *new_end_meta_data_block = (block_end_meta_data_t *)end_meta_data_addr(new_block);
                new_end_meta_data_block->block_size = new_block->block_size;
                insert_node_in_freelist(head,new_block);
            }
            return (void *)((char *)meta_data_block+ALIGN(sizeof(block_meta_data_t)));
        }
    }
    else
    {
        uint32_t req_rem_size = size-meta_data_block->block_size;
        uint32_t inter_frag = 0;
        if(meta_data_block->next_block)
            inter_frag = get_internal_fragmented_size(meta_data_block,meta_data_block->next_block);
        else
            inter_frag = (uint32_t)((char *)mem_heap_hi()+1-((char *)end_meta_data_addr(meta_data_block)+ALIGN(sizeof(block_end_meta_data_t))));

        if(req_rem_size <= inter_frag)
        {
            // TODO check sizeof function once
            meta_data_block->block_size = size;
            block_end_meta_data_t *end_meta_data_block = (block_end_meta_data_t *)end_meta_data_addr(meta_data_block);
            end_meta_data_block->block_size = meta_data_block->block_size;

            return (void *)((char *)meta_data_block+ALIGN(sizeof(block_meta_data_t)));  // Not handling adding remaining free part to free list because it is small in size to handle even headers, so skkipping.
        }

         if(meta_data_block->next_block && meta_data_block->next_block->is_free==MM_TRUE && meta_data_block->prev_block && meta_data_block->prev_block->is_free==MM_TRUE && \
                                                        req_rem_size <= TOTAL_COMBINED_SIZE_OF_BLOCKS(meta_data_block)+inter_frag)
        {
            remove_block_from_free_list(head,&meta_data_block->prev_block->free_node);
            remove_block_from_free_list(head,&meta_data_block->next_block->free_node);
            req_rem_size = abs(meta_data_block->prev_block->block_size-size);
            req_rem_size -= inter_frag;
            inter_frag = get_internal_fragmented_size(meta_data_block->prev_block,meta_data_block);
            req_rem_size -= inter_frag;
            uint32_t final_remain = abs(meta_data_block->next_block->block_size+meta_data_block->block_size+2*ALIGN(sizeof(block_end_meta_data_t)) \
                                                                                                                            +2*ALIGN(sizeof(block_meta_data_t)) - req_rem_size);
            meta_data_block->prev_block->block_size = size;
            meta_data_block = meta_data_block->prev_block;
            block_meta_data_t *store_head = meta_data_block->next_block;
            meta_data_block->next_block = meta_data_block->next_block->next_block->next_block;
            meta_data_block->is_free = MM_FALSE;
            memcpy((char *)(meta_data_block)+ALIGN(sizeof(block_meta_data_t)), (char *)(store_head)+ALIGN(sizeof(block_meta_data_t)),store_head->block_size);
            create_new_block_and_insert_into_freelist(meta_data_block,final_remain);
            return (void *)((char *)meta_data_block+ALIGN(sizeof(block_meta_data_t)));
        }

        if(meta_data_block->next_block && meta_data_block->next_block->is_free == MM_TRUE &&  \
                                                        req_rem_size <= (TOTAL_COMBINE_SIZE_OF_BLOCKS(meta_data_block->next_block)+inter_frag))
        {
            remove_block_from_free_list(head,&meta_data_block->next_block->free_node);
            req_rem_size -= inter_frag;
            uint32_t final_remain = TOTAL_COMBINE_SIZE_OF_BLOCKS(meta_data_block->next_block) - req_rem_size;
            meta_data_block->block_size = size;
            block_meta_data_t *store_head = meta_data_block->next_block;
            meta_data_block->next_block = meta_data_block->next_block->next_block;
            create_new_block_and_insert_into_freelist(meta_data_block,final_remain);
            return (void *)((char *)meta_data_block+ALIGN(sizeof(block_meta_data_t)));
        }


        if(meta_data_block->prev_block && meta_data_block->prev_block->is_free == MM_TRUE && \
                                                        req_rem_size <= (TOTAL_COMBINE_SIZE_OF_BLOCKS(meta_data_block->prev_block)+inter_frag)){
                inter_frag = get_internal_fragmented_size(meta_data_block->prev_block,meta_data_block);
                remove_block_from_free_list(head,&meta_data_block->prev_block->free_node);
                req_rem_size -= inter_frag;
                uint32_t final_remain = abs(TOTAL_COMBINE_SIZE_OF_BLOCKS(meta_data_block->prev_block) - req_rem_size);
                meta_data_block->prev_block->block_size = size;
                meta_data_block = meta_data_block->prev_block;
                meta_data_block->is_free = MM_FALSE;
                block_meta_data_t *store_head = meta_data_block->next_block;
                meta_data_block->next_block = meta_data_block->next_block->next_block;
                memcpy((char *)(meta_data_block)+ALIGN(sizeof(block_meta_data_t)), (char *)(store_head)+ALIGN(sizeof(block_meta_data_t)),store_head->block_size);
                create_new_block_and_insert_into_freelist(meta_data_block,final_remain);
                return (void *)((char *)meta_data_block+ALIGN(sizeof(block_meta_data_t)));
        }


        void *ptr = mm_malloc(size);
        block_meta_data_t *new_meta_data_block = (block_meta_data_t *)((char *)ptr-ALIGN(sizeof(block_meta_data_t)));
        uint32_t amount_to_be_copied = meta_data_block->block_size;
        memcpy((char *)(new_meta_data_block)+ALIGN(sizeof(block_meta_data_t)), (char *)(meta_data_block)+ALIGN(sizeof(block_meta_data_t)),amount_to_be_copied);
        mm_free((void *)((char *)meta_data_block+ALIGN(sizeof(block_meta_data_t))));
        return (void *)((char *)new_meta_data_block+ALIGN(sizeof(block_meta_data_t)));
    }

}

void traverse_list(free_node_t *root)
{
     while(root!=NULL)
     {
        block_meta_data_t *meta_data=meta_data_block_addr(root);

        printf("Block size =%d\n",meta_data->block_size);

        root=root->next_node;
     }

}


// int main(int argc,char *argv[])
// {
//    mm_init();
 
//     void *p1= mm_malloc(30);
//     void *p2 = mm_malloc(20);
//     p1 = mm_realloc(p1,640);
//     traverse_list(*head);
//     printf("[FIRST PART]\n");
//     void *p3 = mm_malloc(128);
//     mm_free(p2);
//     traverse_list(*head);
//     printf("[SECOND PART]\n");


//     p1 = mm_realloc(p1,768);
//     traverse_list(*head);
//     printf("[THIRD PART]\n");
//     void *p4 = mm_malloc(128);
//     mm_free(p3);
//     traverse_list(*head);
//     printf("[FOURTH PART]\n");
//     p1 = mm_realloc(p1,896);
//     void *p5 = mm_malloc(128);
//     block_meta_data_t *meta_data = (block_meta_data_t *)((char *)p1-sizeof(block_meta_data_t));
//     printf("%d %d \n",meta_data->block_size, meta_data->next_block->block_size);
//     mm_free(p4);
//     traverse_list(*head);
//     printf("FIFTH PART\n");


//     p1 = mm_realloc(p1,1024);
//     void *p6 = mm_malloc(128);
//     mm_free(p5);
//     traverse_list(*head);
//     printf("SIXTH PART\n");

//     p1 = mm_realloc(p1,1152);
//     void *p7 = mm_malloc(128);
//     traverse_list(*head);
//     printf("SEVENTH PART\n");
// //    p1=mm_realloc(p1,640);

// //    block_meta_data_t *meta_data = (block_meta_data_t *)((char *)p1-sizeof(block_meta_data_t));

// //    printf("%d %d \n", meta_data->block_size , meta_data->next_block->block_size);


//     mm_free(p6);
//     p1 = mm_realloc(p1,1200);
//     void *p8 = mm_malloc(128);
 
//    traverse_list(*head);
//     printf("EIGHT PART\n");

//     return 0;
// }
