/*-------------------------------------------------------------------------
 *
 * tupmacs.h
 *	  Tuple macros used by both index tuples and heap tuples.
 *
 *
 * Portions Copyright (c) 1996-2001, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 * $Id: tupmacs.h,v 1.18 2001/10/25 05:49:55 momjian Exp $
 *
 *-------------------------------------------------------------------------
 */
#ifndef TUPMACS_H
#define TUPMACS_H

#include "utils/memutils.h"

/*
 * check to see if the ATT'th bit of an array of 8-bit bytes is set.
 */
#define att_isnull(ATT, BITS) (!((BITS)[(ATT) >> 3] & (1 << ((ATT) & 0x07))))

/*
 * Given a Form_pg_attribute and a pointer into a tuple's data area,
 * return the correct value or pointer.
 *
 * We return a Datum value in all cases.  If the attribute has "byval" false,
 * we return the same pointer into the tuple data area that we're passed.
 * Otherwise, we return the correct number of bytes fetched from the data
 * area and extended to Datum form.
 *
 * On machines where Datum is 8 bytes, we support fetching 8-byte byval
 * attributes; otherwise, only 1, 2, and 4-byte values are supported.
 *
 * Note that T must already be properly aligned for this to work correctly.
 */
#define fetchatt(A,T) fetch_att(T, (A)->attbyval, (A)->attlen)

/*
 * Same, but work from byval/len parameters rather than Form_pg_attribute.
 */
#if SIZEOF_DATUM == 8

#define fetch_att(T,attbyval,attlen) \
( \
	(attbyval) ? \
	( \
		(attlen) == (int) sizeof(Datum) ? \
			*((Datum *)(T)) \
		: \
	  ( \
		(attlen) == (int) sizeof(int32) ? \
			Int32GetDatum(*((int32 *)(T))) \
		: \
		( \
			(attlen) == (int) sizeof(int16) ? \
				Int16GetDatum(*((int16 *)(T))) \
			: \
			( \
				AssertMacro((attlen) == 1), \
				CharGetDatum(*((char *)(T))) \
			) \
		) \
	  ) \
	) \
	: \
	PointerGetDatum((char *) (T)) \
)

#else							/* SIZEOF_DATUM != 8 */

#define fetch_att(T,attbyval,attlen) \
( \
	(attbyval) ? \
	( \
		(attlen) == (int) sizeof(int32) ? \
			Int32GetDatum(*((int32 *)(T))) \
		: \
		( \
			(attlen) == (int) sizeof(int16) ? \
				Int16GetDatum(*((int16 *)(T))) \
			: \
			( \
				AssertMacro((attlen) == 1), \
				CharGetDatum(*((char *)(T))) \
			) \
		) \
	) \
	: \
	PointerGetDatum((char *) (T)) \
)
#endif	 /* SIZEOF_DATUM == 8 */

/*
 * att_align aligns the given offset as needed for a datum of length attlen
 * and alignment requirement attalign.	In practice we don't need the length.
 * The attalign cases are tested in what is hopefully something like their
 * frequency of occurrence.
 */
#define att_align(cur_offset, attlen, attalign) \
( \
	((attalign) == 'i') ? INTALIGN(cur_offset) : \
	 (((attalign) == 'c') ? ((long)(cur_offset)) : \
	  (((attalign) == 'd') ? DOUBLEALIGN(cur_offset) : \
		( \
			AssertMacro((attalign) == 's'), \
			SHORTALIGN(cur_offset) \
		))) \
)

/*
 * att_addlength increments the given offset by the length of the attribute.
 * attval is only accessed if we are dealing with a varlena attribute.
 */
#define att_addlength(cur_offset, attlen, attval) \
( \
	((attlen) != -1) ? \
	( \
		(cur_offset) + (attlen) \
	) \
	: \
	( \
		(cur_offset) + VARATT_SIZE(DatumGetPointer(attval)) \
	) \
)

/*
 * store_att_byval is a partial inverse of fetch_att: store a given Datum
 * value into a tuple data area at the specified address.  However, it only
 * handles the byval case, because in typical usage the caller needs to
 * distinguish by-val and by-ref cases anyway, and so a do-it-all macro
 * wouldn't be convenient.
 */
#if SIZEOF_DATUM == 8

#define store_att_byval(T,newdatum,attlen) \
	do { \
		switch (attlen) \
		{ \
			case sizeof(char): \
				*(char *) (T) = DatumGetChar(newdatum); \
				break; \
			case sizeof(int16): \
				*(int16 *) (T) = DatumGetInt16(newdatum); \
				break; \
			case sizeof(int32): \
				*(int32 *) (T) = DatumGetInt32(newdatum); \
				break; \
			case sizeof(Datum): \
				*(Datum *) (T) = (newdatum); \
				break; \
			default: \
				elog(ERROR, "store_att_byval: unsupported byval length %d", \
					 (int) (attlen)); \
				break; \
		} \
	} while (0)

#else							/* SIZEOF_DATUM != 8 */

#define store_att_byval(T,newdatum,attlen) \
	do { \
		switch (attlen) \
		{ \
			case sizeof(char): \
				*(char *) (T) = DatumGetChar(newdatum); \
				break; \
			case sizeof(int16): \
				*(int16 *) (T) = DatumGetInt16(newdatum); \
				break; \
			case sizeof(int32): \
				*(int32 *) (T) = DatumGetInt32(newdatum); \
				break; \
			default: \
				elog(ERROR, "store_att_byval: unsupported byval length %d", \
					 (int) (attlen)); \
				break; \
		} \
	} while (0)
#endif	 /* SIZEOF_DATUM == 8 */
#endif
